/*
 *	argz.c
 *
 *	Argz Functions - Operations on argz vectors.
 *
 *	* Thu Oct 22 21:13:48 JST 2015 Naoyuki Sawa
 *	- 1st [XB
 *	- uArgz Functionsv(http://www.gnu.org/software/libc/manual/html_node/Argz-Functions.html)݊W[łB
 *	  ֐dĺuArgz FunctionsvƓŁA͓ƎłB
 *	  IWił́uArgz Functionsv͑xdĂ╡GłA͒PɎ܂B
 *	- ֐dlg̐́AL̃hLg̑Aȉ̃hLgQlɂȂ܂B
 *	  uLinux Programmer's Manual - ARGZ_ADDv(http://linuxjm.osdn.jp/html/LDP_man-pages/man3/argz_add.3.html)
 *	  AuLinux Programmer's Manual - ARGZ_ADDv́Aargz_append()̐ɊԈႢL悤łB
 *	  ڍׂ́Aargz_append()̎̃RgQƂĉB
 */
#include "clip.h"
/*****************************************************************************
 *	
 *****************************************************************************/
//uThe GNU C Library - 5.12.1 Argz Functionsv(http://www.gnu.org/software/libc/manual/html_node/Argz-Functions.html)p:
//Each argz vector is represented by a pointer to the first element, of type char *, and a size, of type size_t, both of which can be initialized to 0 to represent an empty argz vector.
//All argz functions accept either a pointer and a size argument, or pointers to them, if they will be modified.
//The argz functions use malloc/realloc to allocate/grow argz vectors, and so any argz vector creating using these functions may be freed by using free; conversely, any argz function that may grow a string expects that string to have been allocated using malloc (those argz functions that only examine their arguments or modify them in place will work on any sort of memory).
//See Unconstrained Allocation.
//All argz functions that do memory allocation have a return type of error_t, and return 0 for success, and ENOMEM if an allocation error occurs.
//These functions are declared in the standard include file argz.h.
/*---------------------------------------------------------------------------*/
//The argz_create function converts the Unix-style argument vector argv (a vector of pointers to normal C strings, terminated by (char *)0; see Program Arguments) into an argz vector with the same elements, which is returned in argz and argz_len.
error_t argz_create(char* argv[], char** argz, size_t* argz_len) {
	struct obstack o;
	//ObstackB
	obstack_init(&o);
	//argv̊evfɂāc
	while(*argv) {
		//Obstackɂ̗vf̕'\0'ǉB
		obstack_grow0(&o, *argv, strlen(*argv));
		//argv̗vf֐i߂B
		argv++;
	}
	//vfłL΁c
	if((*argz_len = obstack_object_size(&o))) {
		//IuWFNg̃Rs[B
		*argz = memdup(obstack_base(&o), *argz_len);
		if(!*argz) { DIE(); }
	//vf΁c
	} else {
		//argzNULL|C^ƂB
		*argz = NULL;
	}
	//ObstackJB
	obstack_free(&o, NULL);
	return 0;
}
/*---------------------------------------------------------------------------*/
//The argz_create_sep function converts the null-terminated string string into an argz vector (returned in argz and argz_len) by splitting it into elements at every occurrence of the character sep.
error_t argz_create_sep(const char* str, int sep, char** argz, size_t* argz_len) {
	//Zp[^ꕶ琬镶쐬B
	char _sep[2] = { sep, '\0' };
	//쐬B
	char** argv = strsplit(str, _sep, 0);
	//argv쐬B
	if(argz_create(argv, argz, argz_len)) { DIE(); }
	//JB
	strfreev(argv);
	return 0;
}
/*---------------------------------------------------------------------------*/
//Returns the number of elements in the argz vector argz and argz_len.
size_t argz_count(const char* argz, size_t argz_len) {
	//argzS(Ō'\0'܂)A'\0'̐𐔂B
	size_t count = 0;
	while(argz_len--) {
		if(!*argz++) { count++; }
	}
	return count;
}
/*---------------------------------------------------------------------------*/
//The argz_extract function converts the argz vector argz and argz_len into a Unix-style argument vector stored in argv, by putting pointers to every element in argz into successive positions in argv, followed by a terminator of 0.
//Argv must be pre-allocated with enough space to hold all the elements in argz plus the terminating (char *)0 ((argz_count (argz, argz_len) + 1) * sizeof (char *) bytes should be enough).
//Note that the string pointers stored into argv point into argz-they are not copies-and so argz must be copied if it will be changed while argv is still active.
//This function is useful for passing the elements in argz to an exec function (see Executing a File).
void argz_extract(const char* argz, size_t argz_len, char** argv) {
	//argz̊evfɂāc
	char* iter = NULL;
	while((iter = argz_next(argz, argz_len, iter))) {
		//argvargz̊evfւ̃|C^i[B
		*argv++ = iter;
	}
	//argvɏI[NULL|C^i[B
	*argv = NULL;
}
/*---------------------------------------------------------------------------*/
//The argz_stringify converts argz into a normal string with the elements separated by the character sep, by replacing each '\0' inside argz (except the last one, which terminates the string) with sep.
//This is handy for printing argz in a readable manner.
void argz_stringify(char* argz, size_t argz_len, int sep) {
	//argzȂΉȂB
	// - argzłȂ΍Ō'\0'ΏۊOƂ邽߂ɁAargz_len|XgfNgĂB
	if(argz_len--) {
		//argzS(Ō'\0'͏ΏۊO)A'\0'sepɒuB
		while(argz_len--) {
			if(!*argz) { *argz = sep; }
			argz++;
		}
	}
}
/*---------------------------------------------------------------------------*/
//The argz_add function adds the string str to the end of the argz vector *argz, and updates *argz and *argz_len accordingly.
error_t argz_add(char** argz, size_t* argz_len, const char* str) {
	//vfargz vectorƌȂargz_append()ĂяoB
	return argz_append(argz, argz_len, str, strlen(str) + 1/*nul*/);
}
/*---------------------------------------------------------------------------*/
//The argz_add_sep function is similar to argz_add, but str is split into separate elements in the result at occurrences of the character delim.
//This is useful, for instance, for adding the components of a Unix search path to an argz vector, by using a value of ':' for delim.
error_t argz_add_sep(char** argz, size_t* argz_len, const char* str, int sep) {
	char* buf;
	size_t buf_len;
	//𕪊ĈꎞIargz vector쐬B
	if(argz_create_sep(str, sep, &buf, &buf_len)) { DIE(); }
	//argžɁAꎞIargz vector̓eǉB
	// - uLinux Programmer's Manual - ARGZ_ADDv(http://linuxjm.osdn.jp/html/LDP_man-pages/man3/argz_add.3.html)ɂ́A
	//   argz_append()argz vector(buf,buf_len)̌(*argz,*argz_len)tA*argz*argz_lenXVB
	//   ƏĂ܂AԈႢłB{glibc̃\[XƂA
	//   argz_append()(*argz,*argz_len)̌argz vector(buf,buf_len)tA*argz*argz_lenXVB
	//   ƂɂȂĂ悤łAہA̕RƎv܂B]āAglibcɕāA
	//   argz_append()(*argz,*argz_len)̌argz vector(buf,buf_len)tA*argz*argz_lenXVB
	//   Ƃɂ鎖ɂ܂B
	if(argz_append(argz, argz_len, buf, buf_len)) { DIE(); }
	//ꎞIargz vectorJB
	free(buf);
	return 0;
}
/*---------------------------------------------------------------------------*/
//The argz_append function appends buf_len bytes starting at buf to the argz vector *argz, reallocating *argz to accommodate it, and adding buf_len to *argz_len.
error_t argz_append(char** argz, size_t* argz_len, const char* buf, size_t buf_len) {
	//argz̗eʂAǉargz vector̂ԂAgB
	*argz = realloc(*argz, *argz_len + buf_len);
	if(!*argz) { DIE(); }
	//argžɁAǉargz vector̓eRs[B
	memcpy(*argz + *argz_len, buf, buf_len);
	//argz̃TCYAǉargz vector̂ԂA₷B
	*argz_len += buf_len;
	return 0;
}
/*---------------------------------------------------------------------------*/
//If entry points to the beginning of one of the elements in the argz vector *argz, the argz_delete function will remove this entry and reallocate *argz, modifying *argz and *argz_len accordingly.
//Note that as destructive argz functions usually reallocate their argz argument, pointers into argz vectors such as entry will then become invalid.
void argz_delete(char** argz, size_t* argz_len, char* entry) {
	struct obstack o;
	char* iter;
	//ObstackB
	obstack_init(&o);
	//argz̊evfɂāc
	iter = NULL;
	while((iter = argz_next(*argz, *argz_len, iter))) {
		//̗vf̃|C^Aw肳ꂽ|C^ɈvȂ΁c
		// - ̓eł͂ȂA|C^Ŕr֐dlł鎖ɒӂB
		if(iter != entry) {
			//Obstackɂ̗vf̕'\0'ǉB
			obstack_grow0(&o, iter, strlen(iter));
		}
	}
	//vfłL΁c
	if((*argz_len = obstack_object_size(&o))) {
		//IuWFNg̃Rs[B
		*argz = memdup(obstack_base(&o), *argz_len);
		if(!*argz) { DIE(); }
	//vf΁c
	} else {
		//argzNULL|C^ƂB
		*argz = NULL;
	}
	obstack_free(&o, NULL);
}
/*---------------------------------------------------------------------------*/
//The argz_insert function inserts the string entry into the argz vector *argz at a point just before the existing element pointed to by before, reallocating *argz and updating *argz and *argz_len.
//If before is 0, entry is added to the end instead (as if by argz_add).
//Since the first element is in fact the same as *argz, passing in *argz as the value of before will result in entry being inserted at the beginning.
error_t argz_insert(char** argz, size_t* argz_len, char* before, const char* entry) {
	struct obstack o;
	char* iter;
	if(!before) { return argz_add(argz, argz_len, entry); }
	//ObstackB
	obstack_init(&o);
	//argz̊evfɂāc
	iter = NULL;
	while((iter = argz_next(*argz, *argz_len, iter))) {
		//̗vf̃|C^Aw肳ꂽ|C^Ɉvc
		// - ̓eł͂ȂA|C^Ŕr֐dlł鎖ɒӂB
		if(iter == before) {
			//ObstackɎw肳ꂽ'\0'ǉB
			obstack_grow0(&o, (char*)entry, strlen(entry));	//̃LXg͌x}̂߂łB{obstack_grow0()̑(const void*)łׂȂ̂łobstack_grow0()̊֐dl(void*)ƒ`Ă̂Ŏ~ނ𓾂ĂяoŃLXgČx}鎖ɂ܂B
		}
		//Obstackɂ̗vf̕'\0'ǉB
		obstack_grow0(&o, iter, strlen(iter));
	}
	//vfłL΁c
	if((*argz_len = obstack_object_size(&o))) {
		//IuWFNg̃Rs[B
		*argz = memdup(obstack_base(&o), *argz_len);
		if(!*argz) { DIE(); }
	//vf΁c
	} else {
		//argzNULL|C^ƂB
		*argz = NULL;
	}
	//ObstackJB
	obstack_free(&o, NULL);
	return 0;
}
/*---------------------------------------------------------------------------*/
//The argz_next function provides a convenient way of iterating over the elements in the argz vector argz.
//It returns a pointer to the next element in argz after the element entry, or 0 if there are no elements following entry.
//If entry is 0, the first element of argz is returned.
//This behavior suggests two styles of iteration:
//char *entry = 0;
//while ((entry = argz_next (argz, argz_len, entry)))
//  action;
//(the double parentheses are necessary to make some C compilers shut up about what they consider a questionable while-test) and:
//char *entry;
//for (entry = argz;
//     entry;
//     entry = argz_next (argz, argz_len, entry))
//  action;
//Note that the latter depends on argz having a value of 0 if it is empty (rather than a pointer to an empty block of memory); this invariant is maintained for argz vectors created by the functions here.
char* argz_next(const char* argz, size_t argz_len, const char* entry) {
	//NULL|C^w肳ꂽAargz̐擪vfւ̃|C^ԂB
	// - argzȂ(argz==NULL)Ȃ̂ŁAɂĂargẑ܂ܕԂΗǂB
	if(!entry) { return (char*)argz; }
	//̗vfւ̃|C^߂B
	entry += strlen(entry) + 1/*nul*/;
	//̗vf,,argz̖ɒBNULL|C^ԂB
	return (entry < argz + argz_len) ? (char*)entry : NULL;
}
/*---------------------------------------------------------------------------*/
//Replace any occurrences of the string str in argz with with, reallocating argz as necessary.
//If replace_count is non-zero, *replace_count will be incremented by number of replacements performed.
error_t argz_replace(char** argz, size_t* argz_len, const char* str, const char* with, unsigned* replace_count) {
	struct obstack o;
	char* iter;
	//ObstackB
	obstack_init(&o);
	//argz̊evfɂāc
	iter = NULL;
	while((iter = argz_next(*argz, *argz_len, iter))) {
		//̗vf̕񂪁Aw肳ꂽɈvȂ΁c
		// - |C^ł͂ȂA̓eŔr֐dlł鎖ɒӂB
		if(strcmp(iter, str)) {
			//Obstackɂ̗vf̕'\0'ǉB
			obstack_grow0(&o, iter, strlen(iter));
		//̗vf̕񂪁Aw肳ꂽɈvc
		} else {
			//ObstackɎw肳ꂽ'\0'ǉB
			obstack_grow0(&o, (char*)with, strlen(with));	//̃LXg͌x}̂߂łB{obstack_grow0()̑(const void*)łׂȂ̂łobstack_grow0()̊֐dl(void*)ƒ`Ă̂Ŏ~ނ𓾂ĂяoŃLXgČx}鎖ɂ܂B
			//u񐔂̕ϐ|C^w肳ĂAu񐔂𑝂₷B
			// - ֐(*replace_count)ȂɒӂB₷Ƃ֐dlłB
			if(replace_count) { (*replace_count)++; }
		}
	}
	//vfłL΁c
	if((*argz_len = obstack_object_size(&o))) {
		//IuWFNg̃Rs[B
		*argz = memdup(obstack_base(&o), *argz_len);
		if(!*argz) { DIE(); }
	//vf΁c
	} else {
		//argzNULL|C^ƂB
		*argz = NULL;
	}
	//ObstackJB
	obstack_free(&o, NULL);
	return 0;
}
