/*	
 *	clipxm.c
 *
 *	P/ECE XM Driver
 *
 *	CLiP - Common Library for P/ECE
 *	Copyright (C) 2001-2004 Naoyuki Sawa
 *
 *	* Tue Nov 9 19:00:00 JST 2004 Naoyuki Sawa
 *	- 쐬JnB
 */
#include "clip.h"

/* * G[R[h̒ĺAsԍ𕉒lɂ̂łB
 *   vOCƁAG[R[hς\܂B
 *   G[R[h̒ĺAfobÔ߂ɗpĂB
 */
#define ERRNO	(-__LINE__)

/****************************************************************************
 *	P/ECE XM Driver
 ****************************************************************************/

/* * period = (note << 8) + (finetune << 1) + (porta up/down);
 *   frequency = xm_linear_frequency_table[period % (12 * 256)]
 *				       << (period / (12 * 256));
 * * {́APorta up/down̒Pʂ͔2551(=1IN^[u(12*255)1)Ȃ̂łA
 *   ȒP̂߂ɁA2561(=1IN^[u(12*256)1)Ƃ܂B
 *   ܂AႢ͂킩ȂƎv܂B
 */
const short xm_linear_frequency_table[12 * 256] = {
/* 8363/16*2^(( 0/12)*256).. */
 523, 523, 523, 523, 523, 523, 523, 524, 524, 524, 524, 524, 524, 524, 524, 524, 525, 525, 525, 525, 525, 525, 525, 525, 526, 526, 526, 526, 526, 526, 526, 526,
 526, 527, 527, 527, 527, 527, 527, 527, 527, 528, 528, 528, 528, 528, 528, 528, 528, 528, 529, 529, 529, 529, 529, 529, 529, 529, 530, 530, 530, 530, 530, 530,
 530, 530, 531, 531, 531, 531, 531, 531, 531, 531, 531, 532, 532, 532, 532, 532, 532, 532, 532, 533, 533, 533, 533, 533, 533, 533, 533, 534, 534, 534, 534, 534,
 534, 534, 534, 534, 535, 535, 535, 535, 535, 535, 535, 535, 536, 536, 536, 536, 536, 536, 536, 536, 537, 537, 537, 537, 537, 537, 537, 537, 538, 538, 538, 538,
 538, 538, 538, 538, 538, 539, 539, 539, 539, 539, 539, 539, 539, 540, 540, 540, 540, 540, 540, 540, 540, 541, 541, 541, 541, 541, 541, 541, 541, 542, 542, 542,
 542, 542, 542, 542, 542, 543, 543, 543, 543, 543, 543, 543, 543, 543, 544, 544, 544, 544, 544, 544, 544, 544, 545, 545, 545, 545, 545, 545, 545, 545, 546, 546,
 546, 546, 546, 546, 546, 546, 547, 547, 547, 547, 547, 547, 547, 547, 548, 548, 548, 548, 548, 548, 548, 548, 549, 549, 549, 549, 549, 549, 549, 549, 550, 550,
 550, 550, 550, 550, 550, 550, 551, 551, 551, 551, 551, 551, 551, 551, 552, 552, 552, 552, 552, 552, 552, 552, 553, 553, 553, 553, 553, 553, 553, 553, 554, 554,
/* 8363/16*2^(( 1/12)*256).. */
 554, 554, 554, 554, 554, 554, 555, 555, 555, 555, 555, 555, 555, 555, 556, 556, 556, 556, 556, 556, 556, 556, 557, 557, 557, 557, 557, 557, 557, 557, 558, 558,
 558, 558, 558, 558, 558, 558, 559, 559, 559, 559, 559, 559, 559, 559, 560, 560, 560, 560, 560, 560, 560, 560, 561, 561, 561, 561, 561, 561, 561, 561, 562, 562,
 562, 562, 562, 562, 562, 562, 563, 563, 563, 563, 563, 563, 563, 563, 564, 564, 564, 564, 564, 564, 564, 564, 565, 565, 565, 565, 565, 565, 565, 566, 566, 566,
 566, 566, 566, 566, 566, 567, 567, 567, 567, 567, 567, 567, 567, 568, 568, 568, 568, 568, 568, 568, 568, 569, 569, 569, 569, 569, 569, 569, 569, 570, 570, 570,
 570, 570, 570, 570, 571, 571, 571, 571, 571, 571, 571, 571, 572, 572, 572, 572, 572, 572, 572, 572, 573, 573, 573, 573, 573, 573, 573, 573, 574, 574, 574, 574,
 574, 574, 574, 575, 575, 575, 575, 575, 575, 575, 575, 576, 576, 576, 576, 576, 576, 576, 576, 577, 577, 577, 577, 577, 577, 577, 578, 578, 578, 578, 578, 578,
 578, 578, 579, 579, 579, 579, 579, 579, 579, 579, 580, 580, 580, 580, 580, 580, 580, 581, 581, 581, 581, 581, 581, 581, 581, 582, 582, 582, 582, 582, 582, 582,
 582, 583, 583, 583, 583, 583, 583, 583, 584, 584, 584, 584, 584, 584, 584, 584, 585, 585, 585, 585, 585, 585, 585, 586, 586, 586, 586, 586, 586, 586, 586, 587,
/* 8363/16*2^(( 2/12)*256).. */
 587, 587, 587, 587, 587, 587, 587, 588, 588, 588, 588, 588, 588, 588, 589, 589, 589, 589, 589, 589, 589, 589, 590, 590, 590, 590, 590, 590, 590, 591, 591, 591,
 591, 591, 591, 591, 591, 592, 592, 592, 592, 592, 592, 592, 593, 593, 593, 593, 593, 593, 593, 593, 594, 594, 594, 594, 594, 594, 594, 595, 595, 595, 595, 595,
 595, 595, 595, 596, 596, 596, 596, 596, 596, 596, 597, 597, 597, 597, 597, 597, 597, 598, 598, 598, 598, 598, 598, 598, 598, 599, 599, 599, 599, 599, 599, 599,
 600, 600, 600, 600, 600, 600, 600, 600, 601, 601, 601, 601, 601, 601, 601, 602, 602, 602, 602, 602, 602, 602, 603, 603, 603, 603, 603, 603, 603, 603, 604, 604,
 604, 604, 604, 604, 604, 605, 605, 605, 605, 605, 605, 605, 606, 606, 606, 606, 606, 606, 606, 606, 607, 607, 607, 607, 607, 607, 607, 608, 608, 608, 608, 608,
 608, 608, 609, 609, 609, 609, 609, 609, 609, 610, 610, 610, 610, 610, 610, 610, 610, 611, 611, 611, 611, 611, 611, 611, 612, 612, 612, 612, 612, 612, 612, 613,
 613, 613, 613, 613, 613, 613, 614, 614, 614, 614, 614, 614, 614, 614, 615, 615, 615, 615, 615, 615, 615, 616, 616, 616, 616, 616, 616, 616, 617, 617, 617, 617,
 617, 617, 617, 618, 618, 618, 618, 618, 618, 618, 619, 619, 619, 619, 619, 619, 619, 619, 620, 620, 620, 620, 620, 620, 620, 621, 621, 621, 621, 621, 621, 621,
/* 8363/16*2^(( 3/12)*256).. */
 622, 622, 622, 622, 622, 622, 622, 623, 623, 623, 623, 623, 623, 623, 624, 624, 624, 624, 624, 624, 624, 625, 625, 625, 625, 625, 625, 625, 626, 626, 626, 626,
 626, 626, 626, 627, 627, 627, 627, 627, 627, 627, 628, 628, 628, 628, 628, 628, 628, 628, 629, 629, 629, 629, 629, 629, 629, 630, 630, 630, 630, 630, 630, 630,
 631, 631, 631, 631, 631, 631, 631, 632, 632, 632, 632, 632, 632, 632, 633, 633, 633, 633, 633, 633, 633, 634, 634, 634, 634, 634, 634, 634, 635, 635, 635, 635,
 635, 635, 635, 636, 636, 636, 636, 636, 636, 636, 637, 637, 637, 637, 637, 637, 637, 638, 638, 638, 638, 638, 638, 638, 639, 639, 639, 639, 639, 639, 640, 640,
 640, 640, 640, 640, 640, 641, 641, 641, 641, 641, 641, 641, 642, 642, 642, 642, 642, 642, 642, 643, 643, 643, 643, 643, 643, 643, 644, 644, 644, 644, 644, 644,
 644, 645, 645, 645, 645, 645, 645, 645, 646, 646, 646, 646, 646, 646, 646, 647, 647, 647, 647, 647, 647, 647, 648, 648, 648, 648, 648, 648, 649, 649, 649, 649,
 649, 649, 649, 650, 650, 650, 650, 650, 650, 650, 651, 651, 651, 651, 651, 651, 651, 652, 652, 652, 652, 652, 652, 652, 653, 653, 653, 653, 653, 653, 654, 654,
 654, 654, 654, 654, 654, 655, 655, 655, 655, 655, 655, 655, 656, 656, 656, 656, 656, 656, 656, 657, 657, 657, 657, 657, 657, 658, 658, 658, 658, 658, 658, 658,
/* 8363/16*2^(( 4/12)*256).. */
 659, 659, 659, 659, 659, 659, 659, 660, 660, 660, 660, 660, 660, 660, 661, 661, 661, 661, 661, 661, 662, 662, 662, 662, 662, 662, 662, 663, 663, 663, 663, 663,
 663, 663, 664, 664, 664, 664, 664, 664, 665, 665, 665, 665, 665, 665, 665, 666, 666, 666, 666, 666, 666, 666, 667, 667, 667, 667, 667, 667, 668, 668, 668, 668,
 668, 668, 668, 669, 669, 669, 669, 669, 669, 669, 670, 670, 670, 670, 670, 670, 671, 671, 671, 671, 671, 671, 671, 672, 672, 672, 672, 672, 672, 673, 673, 673,
 673, 673, 673, 673, 674, 674, 674, 674, 674, 674, 674, 675, 675, 675, 675, 675, 675, 676, 676, 676, 676, 676, 676, 676, 677, 677, 677, 677, 677, 677, 678, 678,
 678, 678, 678, 678, 678, 679, 679, 679, 679, 679, 679, 680, 680, 680, 680, 680, 680, 680, 681, 681, 681, 681, 681, 681, 682, 682, 682, 682, 682, 682, 682, 683,
 683, 683, 683, 683, 683, 684, 684, 684, 684, 684, 684, 684, 685, 685, 685, 685, 685, 685, 686, 686, 686, 686, 686, 686, 686, 687, 687, 687, 687, 687, 687, 688,
 688, 688, 688, 688, 688, 688, 689, 689, 689, 689, 689, 689, 690, 690, 690, 690, 690, 690, 690, 691, 691, 691, 691, 691, 691, 692, 692, 692, 692, 692, 692, 693,
 693, 693, 693, 693, 693, 693, 694, 694, 694, 694, 694, 694, 695, 695, 695, 695, 695, 695, 696, 696, 696, 696, 696, 696, 696, 697, 697, 697, 697, 697, 697, 698,
/* 8363/16*2^(( 5/12)*256).. */
 698, 698, 698, 698, 698, 698, 699, 699, 699, 699, 699, 699, 700, 700, 700, 700, 700, 700, 701, 701, 701, 701, 701, 701, 701, 702, 702, 702, 702, 702, 702, 703,
 703, 703, 703, 703, 703, 704, 704, 704, 704, 704, 704, 705, 705, 705, 705, 705, 705, 705, 706, 706, 706, 706, 706, 706, 707, 707, 707, 707, 707, 707, 708, 708,
 708, 708, 708, 708, 708, 709, 709, 709, 709, 709, 709, 710, 710, 710, 710, 710, 710, 711, 711, 711, 711, 711, 711, 712, 712, 712, 712, 712, 712, 712, 713, 713,
 713, 713, 713, 713, 714, 714, 714, 714, 714, 714, 715, 715, 715, 715, 715, 715, 716, 716, 716, 716, 716, 716, 717, 717, 717, 717, 717, 717, 718, 718, 718, 718,
 718, 718, 718, 719, 719, 719, 719, 719, 719, 720, 720, 720, 720, 720, 720, 721, 721, 721, 721, 721, 721, 722, 722, 722, 722, 722, 722, 723, 723, 723, 723, 723,
 723, 724, 724, 724, 724, 724, 724, 724, 725, 725, 725, 725, 725, 725, 726, 726, 726, 726, 726, 726, 727, 727, 727, 727, 727, 727, 728, 728, 728, 728, 728, 728,
 729, 729, 729, 729, 729, 729, 730, 730, 730, 730, 730, 730, 731, 731, 731, 731, 731, 731, 732, 732, 732, 732, 732, 732, 733, 733, 733, 733, 733, 733, 734, 734,
 734, 734, 734, 734, 735, 735, 735, 735, 735, 735, 736, 736, 736, 736, 736, 736, 737, 737, 737, 737, 737, 737, 738, 738, 738, 738, 738, 738, 739, 739, 739, 739,
/* 8363/16*2^(( 6/12)*256).. */
 739, 739, 740, 740, 740, 740, 740, 740, 741, 741, 741, 741, 741, 741, 742, 742, 742, 742, 742, 742, 743, 743, 743, 743, 743, 743, 744, 744, 744, 744, 744, 744,
 745, 745, 745, 745, 745, 745, 746, 746, 746, 746, 746, 746, 747, 747, 747, 747, 747, 747, 748, 748, 748, 748, 748, 748, 749, 749, 749, 749, 749, 749, 750, 750,
 750, 750, 750, 750, 751, 751, 751, 751, 751, 751, 752, 752, 752, 752, 752, 752, 753, 753, 753, 753, 753, 754, 754, 754, 754, 754, 754, 755, 755, 755, 755, 755,
 755, 756, 756, 756, 756, 756, 756, 757, 757, 757, 757, 757, 757, 758, 758, 758, 758, 758, 758, 759, 759, 759, 759, 759, 759, 760, 760, 760, 760, 760, 761, 761,
 761, 761, 761, 761, 762, 762, 762, 762, 762, 762, 763, 763, 763, 763, 763, 763, 764, 764, 764, 764, 764, 764, 765, 765, 765, 765, 765, 766, 766, 766, 766, 766,
 766, 767, 767, 767, 767, 767, 767, 768, 768, 768, 768, 768, 768, 769, 769, 769, 769, 769, 769, 770, 770, 770, 770, 770, 771, 771, 771, 771, 771, 771, 772, 772,
 772, 772, 772, 772, 773, 773, 773, 773, 773, 773, 774, 774, 774, 774, 774, 775, 775, 775, 775, 775, 775, 776, 776, 776, 776, 776, 776, 777, 777, 777, 777, 777,
 778, 778, 778, 778, 778, 778, 779, 779, 779, 779, 779, 779, 780, 780, 780, 780, 780, 781, 781, 781, 781, 781, 781, 782, 782, 782, 782, 782, 782, 783, 783, 783,
/* 8363/16*2^(( 7/12)*256).. */
 783, 783, 783, 784, 784, 784, 784, 784, 785, 785, 785, 785, 785, 785, 786, 786, 786, 786, 786, 787, 787, 787, 787, 787, 787, 788, 788, 788, 788, 788, 788, 789,
 789, 789, 789, 789, 790, 790, 790, 790, 790, 790, 791, 791, 791, 791, 791, 791, 792, 792, 792, 792, 792, 793, 793, 793, 793, 793, 793, 794, 794, 794, 794, 794,
 795, 795, 795, 795, 795, 795, 796, 796, 796, 796, 796, 797, 797, 797, 797, 797, 797, 798, 798, 798, 798, 798, 798, 799, 799, 799, 799, 799, 800, 800, 800, 800,
 800, 800, 801, 801, 801, 801, 801, 802, 802, 802, 802, 802, 802, 803, 803, 803, 803, 803, 804, 804, 804, 804, 804, 804, 805, 805, 805, 805, 805, 806, 806, 806,
 806, 806, 806, 807, 807, 807, 807, 807, 808, 808, 808, 808, 808, 808, 809, 809, 809, 809, 809, 810, 810, 810, 810, 810, 810, 811, 811, 811, 811, 811, 812, 812,
 812, 812, 812, 812, 813, 813, 813, 813, 813, 814, 814, 814, 814, 814, 815, 815, 815, 815, 815, 815, 816, 816, 816, 816, 816, 817, 817, 817, 817, 817, 817, 818,
 818, 818, 818, 818, 819, 819, 819, 819, 819, 819, 820, 820, 820, 820, 820, 821, 821, 821, 821, 821, 822, 822, 822, 822, 822, 822, 823, 823, 823, 823, 823, 824,
 824, 824, 824, 824, 824, 825, 825, 825, 825, 825, 826, 826, 826, 826, 826, 827, 827, 827, 827, 827, 827, 828, 828, 828, 828, 828, 829, 829, 829, 829, 829, 830,
/* 8363/16*2^(( 8/12)*256).. */
 830, 830, 830, 830, 830, 831, 831, 831, 831, 831, 832, 832, 832, 832, 832, 833, 833, 833, 833, 833, 833, 834, 834, 834, 834, 834, 835, 835, 835, 835, 835, 836,
 836, 836, 836, 836, 836, 837, 837, 837, 837, 837, 838, 838, 838, 838, 838, 839, 839, 839, 839, 839, 840, 840, 840, 840, 840, 840, 841, 841, 841, 841, 841, 842,
 842, 842, 842, 842, 843, 843, 843, 843, 843, 843, 844, 844, 844, 844, 844, 845, 845, 845, 845, 845, 846, 846, 846, 846, 846, 847, 847, 847, 847, 847, 848, 848,
 848, 848, 848, 848, 849, 849, 849, 849, 849, 850, 850, 850, 850, 850, 851, 851, 851, 851, 851, 852, 852, 852, 852, 852, 852, 853, 853, 853, 853, 853, 854, 854,
 854, 854, 854, 855, 855, 855, 855, 855, 856, 856, 856, 856, 856, 857, 857, 857, 857, 857, 858, 858, 858, 858, 858, 858, 859, 859, 859, 859, 859, 860, 860, 860,
 860, 860, 861, 861, 861, 861, 861, 862, 862, 862, 862, 862, 863, 863, 863, 863, 863, 864, 864, 864, 864, 864, 864, 865, 865, 865, 865, 865, 866, 866, 866, 866,
 866, 867, 867, 867, 867, 867, 868, 868, 868, 868, 868, 869, 869, 869, 869, 869, 870, 870, 870, 870, 870, 871, 871, 871, 871, 871, 872, 872, 872, 872, 872, 873,
 873, 873, 873, 873, 874, 874, 874, 874, 874, 875, 875, 875, 875, 875, 875, 876, 876, 876, 876, 876, 877, 877, 877, 877, 877, 878, 878, 878, 878, 878, 879, 879,
/* 8363/16*2^(( 9/12)*256).. */
 879, 879, 879, 880, 880, 880, 880, 880, 881, 881, 881, 881, 881, 882, 882, 882, 882, 882, 883, 883, 883, 883, 883, 884, 884, 884, 884, 884, 885, 885, 885, 885,
 885, 886, 886, 886, 886, 886, 887, 887, 887, 887, 887, 888, 888, 888, 888, 888, 889, 889, 889, 889, 889, 890, 890, 890, 890, 890, 891, 891, 891, 891, 891, 892,
 892, 892, 892, 892, 893, 893, 893, 893, 893, 894, 894, 894, 894, 894, 895, 895, 895, 895, 895, 896, 896, 896, 896, 896, 897, 897, 897, 897, 897, 898, 898, 898,
 898, 899, 899, 899, 899, 899, 900, 900, 900, 900, 900, 901, 901, 901, 901, 901, 902, 902, 902, 902, 902, 903, 903, 903, 903, 903, 904, 904, 904, 904, 904, 905,
 905, 905, 905, 905, 906, 906, 906, 906, 906, 907, 907, 907, 907, 907, 908, 908, 908, 908, 908, 909, 909, 909, 909, 910, 910, 910, 910, 910, 911, 911, 911, 911,
 911, 912, 912, 912, 912, 912, 913, 913, 913, 913, 913, 914, 914, 914, 914, 914, 915, 915, 915, 915, 915, 916, 916, 916, 916, 917, 917, 917, 917, 917, 918, 918,
 918, 918, 918, 919, 919, 919, 919, 919, 920, 920, 920, 920, 920, 921, 921, 921, 921, 921, 922, 922, 922, 922, 923, 923, 923, 923, 923, 924, 924, 924, 924, 924,
 925, 925, 925, 925, 925, 926, 926, 926, 926, 927, 927, 927, 927, 927, 928, 928, 928, 928, 928, 929, 929, 929, 929, 929, 930, 930, 930, 930, 930, 931, 931, 931,
/* 8363/16*2^((10/12)*256).. */
 931, 932, 932, 932, 932, 932, 933, 933, 933, 933, 933, 934, 934, 934, 934, 934, 935, 935, 935, 935, 936, 936, 936, 936, 936, 937, 937, 937, 937, 937, 938, 938,
 938, 938, 938, 939, 939, 939, 939, 940, 940, 940, 940, 940, 941, 941, 941, 941, 941, 942, 942, 942, 942, 943, 943, 943, 943, 943, 944, 944, 944, 944, 944, 945,
 945, 945, 945, 946, 946, 946, 946, 946, 947, 947, 947, 947, 947, 948, 948, 948, 948, 949, 949, 949, 949, 949, 950, 950, 950, 950, 950, 951, 951, 951, 951, 952,
 952, 952, 952, 952, 953, 953, 953, 953, 953, 954, 954, 954, 954, 955, 955, 955, 955, 955, 956, 956, 956, 956, 956, 957, 957, 957, 957, 958, 958, 958, 958, 958,
 959, 959, 959, 959, 959, 960, 960, 960, 960, 961, 961, 961, 961, 961, 962, 962, 962, 962, 963, 963, 963, 963, 963, 964, 964, 964, 964, 964, 965, 965, 965, 965,
 966, 966, 966, 966, 966, 967, 967, 967, 967, 968, 968, 968, 968, 968, 969, 969, 969, 969, 969, 970, 970, 970, 970, 971, 971, 971, 971, 971, 972, 972, 972, 972,
 973, 973, 973, 973, 973, 974, 974, 974, 974, 975, 975, 975, 975, 975, 976, 976, 976, 976, 977, 977, 977, 977, 977, 978, 978, 978, 978, 978, 979, 979, 979, 979,
 980, 980, 980, 980, 980, 981, 981, 981, 981, 982, 982, 982, 982, 982, 983, 983, 983, 983, 984, 984, 984, 984, 984, 985, 985, 985, 985, 986, 986, 986, 986, 986,
/* 8363/16*2^((11/12)*256).. */
 987, 987, 987, 987, 988, 988, 988, 988, 988, 989, 989, 989, 989, 990, 990, 990, 990, 990, 991, 991, 991, 991, 992, 992, 992, 992, 993, 993, 993, 993, 993, 994,
 994, 994, 994, 995, 995, 995, 995, 995, 996, 996, 996, 996, 997, 997, 997, 997, 997, 998, 998, 998, 998, 999, 999, 999, 999, 999,1000,1000,1000,1000,1001,1001,
1001,1001,1002,1002,1002,1002,1002,1003,1003,1003,1003,1004,1004,1004,1004,1004,1005,1005,1005,1005,1006,1006,1006,1006,1006,1007,1007,1007,1007,1008,1008,1008,
1008,1009,1009,1009,1009,1009,1010,1010,1010,1010,1011,1011,1011,1011,1011,1012,1012,1012,1012,1013,1013,1013,1013,1014,1014,1014,1014,1014,1015,1015,1015,1015,
1016,1016,1016,1016,1017,1017,1017,1017,1017,1018,1018,1018,1018,1019,1019,1019,1019,1020,1020,1020,1020,1020,1021,1021,1021,1021,1022,1022,1022,1022,1023,1023,
1023,1023,1023,1024,1024,1024,1024,1025,1025,1025,1025,1026,1026,1026,1026,1026,1027,1027,1027,1027,1028,1028,1028,1028,1029,1029,1029,1029,1029,1030,1030,1030,
1030,1031,1031,1031,1031,1032,1032,1032,1032,1032,1033,1033,1033,1033,1034,1034,1034,1034,1035,1035,1035,1035,1036,1036,1036,1036,1036,1037,1037,1037,1037,1038,
1038,1038,1038,1039,1039,1039,1039,1039,1040,1040,1040,1040,1041,1041,1041,1041,1042,1042,1042,1042,1043,1043,1043,1043,1043,1044,1044,1044,1044,1045,1045,1045,
};

/* XM Driver܂B
 * [in]
 *	driver		XMDRIVER\́B
 *	data		XMf[^̐擪AhXB
 * [out]
 *	߂l		Ȃ0Ԃ܂B
 *			sȂ0ȊO̒lԂ܂B
 * [note]
 *	* xm_init()痘p܂B
 *	  ̊֐쐬ŔAsxm_free()ĂяoȒPɂ邽߂łB
 *	  xm_init()̒ɒ_xm_init()̓eĂ܂ƁAso
 *	  ꂼxm_free()ĂяoǉȂ΂ȂȂ܂B
 *	  _xm_init()ЂƂԂɋނƂŁAsoƂxm_free()Ăяo
 *	  xm_init()̒̈ꂩōςނ悤ɂȂ܂B
 */
static int
_xm_init(XMDRIVER* driver, const void* data)
{
	const XMGENERALHEADER* general_header;
	const XMPATTERNHEADER* pattern_header;
	const XMINSTRUMENTHEADER* instrument_header;
	const XMSAMPLEHEADER* sample_header;
	XMPATTERN* pattern;
	XMINSTRUMENT* instrument;
	XMSAMPLE* sample;
	XMCHANNEL* channel;
	int note;
	//int order;
	int i_pattern;
	int i_instrument;
	int i_point;
	int i_sample;
	int i_channel;
	int volume_type;
	int number_of_volume_points;
	int volume_sustain_point;
	int volume_loop_start_point;
	int volume_loop_end_point;
	int sample_length;
	int sample_loop_start;
	int sample_loop_length;
	int sample_header_size;
	const unsigned char* sample_data;
	const unsigned char* sample_position;

	/* ܂AXMDRIVER\̂NA܂B */
	memset(driver, 0, sizeof(XMDRIVER));

	/********************
	 *  General header  *
	 ********************/

	general_header = (const XMGENERALHEADER*)data;

	/* VOl`ƃo[W܂B */
	if(memcmp(general_header->id_text, "Extended Module: ", 17) != 0) {
		return ERRNO;
	}
	if(LEBYTE(general_header->_1a) != 0x1a) {
		return ERRNO;
	}
	if(LEHALF(general_header->version_number) < 0x0103) {
		return ERRNO;
	}

	/* Pattern order table̒擾܂B */
	driver->song_length = LEHALF(general_header->song_length);
	if((driver->song_length < 1) ||
	   (driver->song_length > 256)) {
		return ERRNO;
	}

	/* [vJnʒu擾܂B */
	driver->restart_position = LEHALF(general_header->restart_position);
	//if((driver->restart_position < 0) ||
	//   (driver->restart_position > driver->song_length - 1)) {
	//	return ERRNO;
	//}
	//[vȂȂ́A͈͊ÕI[_[ŏI[ꍇ悤łB
	//  ł̓G[AtɌo邱Ƃɂ܂B

	/* `l擾܂B */
	driver->number_of_channels = LEHALF(general_header->number_of_channels);
	if((driver->number_of_channels & 1) ||
	   (driver->number_of_channels < 2) ||
	   (driver->number_of_channels > 32)) {
		return ERRNO; /* `l2`32̋łȂ΂܂ */
	}

	/* XMCHANNEL\̂mۂ܂B */
	driver->channels = calloc(driver->number_of_channels, sizeof(XMCHANNEL));
	if(!driver->channels) {
		return ERRNO;
	}

	/* Pattern̐擾܂B */
	driver->number_of_patterns = LEHALF(general_header->number_of_patterns);
	if((driver->number_of_patterns < 1) ||
	   (driver->number_of_patterns > 256)) {
		return ERRNO;
	}

	/* Instrument̐擾܂B */
	driver->number_of_instruments = LEHALF(general_header->number_of_instruments);
	if((driver->number_of_instruments < 1) ||
	   (driver->number_of_instruments > 128)) {
		return ERRNO;
	}

	/* Flags擾܂B(ge[u`)
	 * 2004/11/11݁ALinear frequency tablêݑΉBAmiga frequency table͖ΉłB
	 */
	driver->flags = LEHALF(general_header->flags);
	//if(!(driver->flags & 1)) {
	//	return ERRNO;
	//}
	//Amiga frequency tablegXMt@CALinear frequency tablegĉtĂ܂Ƃɂ܂B
	//  ɂ͊ԈႢłAςƒł́A܂Ⴂ͂킩Ȃ悤łB
	//  AAmiga frequency tableɐΉƂ̂߂ɁAAflags͊i[ĂƂɂ܂B
	//  2004/11/13݂́Aflags͂ǂQƂĂ܂B

	/* TempoBPM̏l擾܂B */
	driver->tempo = LEHALF(general_header->default_tempo);
	if(driver->tempo < 1) {
		//return ERRNO;
		//Amiga frequency tablegXMt@C(MOD̃Ro[g?)ł́ADefault tempo=0ƂȂĂꍇ悤łB
		//Default tempoȏꍇ́AMODɂDefault tempo(MODłSpeedɑ)6i[Ă܂B(Ǝf)
		driver->tempo = 6;
	}
	driver->bpm = LEHALF(general_header->default_bpm);
	if(driver->bpm < 1) {
		//return ERRNO;
		//Amiga frequency tablegXMt@CłABPM(MODłTempoɑ)͗LȒlݒ肳Ăꍇ悤łB
		//͂Default tempȍꍇɕāAȏꍇ́AMODɂDefault BPM125i[Ă܂B(Ǝf)
		driver->bpm = 125;
	}

	/* Pattern order table擾܂B */
	driver->pattern_order_table = general_header->pattern_order_table;
	//for(order = 0; order < driver->song_length; order++) {
	//	i_pattern = driver->pattern_order_table[order];
	//	if((i_pattern < 0) ||
	//	   (i_pattern > driver->number_of_patterns - 1)) {
	//		return ERRNO;
	//	}
	//}
	//[vȂȂ́A͈͊Õp^[ԍŏI[ꍇ悤łB
	//  ł̓G[AtɌo邱Ƃɂ܂B

	/**************
	 *  Patterns  *
	 **************/

	pattern_header = (const XMPATTERNHEADER*)((int)general_header->header_size +
						LEWORD(general_header->header_size));

	/* XMPATTERN\̂mۂ܂B */
	driver->patterns = calloc(driver->number_of_patterns, sizeof(XMPATTERN));
	if(!driver->patterns) {
		return ERRNO;
	}

	/* ePatternɂ... */
	for(i_pattern = 0, pattern = driver->patterns;
	    i_pattern < driver->number_of_patterns;
	    i_pattern++, pattern++) {

		/* Row̐擾܂B */
		pattern->number_of_rows_in_pattern = LEHALF(pattern_header->number_of_rows_in_pattern);
		if((pattern->number_of_rows_in_pattern < 1) ||
		   (pattern->number_of_rows_in_pattern > 256)) {
			return ERRNO;
		}

		/* Packed pattern data擾܂B */
		pattern->packed_pattern_data = (const unsigned char*)((int)pattern_header +
								    LEWORD(pattern_header->pattern_header_length));

		/* Pattern֐i߂܂B */
		pattern_header = (const XMPATTERNHEADER*)((int)pattern->packed_pattern_data +
							LEHALF(pattern_header->packed_pattern_data_size));
	}

	/*****************
	 *  Instruments  *
	 *****************/

	instrument_header = (const XMINSTRUMENTHEADER*)pattern_header;

	/* XMINSTRUMENT\̂mۂ܂B */
	driver->instruments = calloc(driver->number_of_instruments, sizeof(XMINSTRUMENT));
	if(!driver->instruments) {
		return ERRNO;
	}

	/* eInstrumentɂ... */
	for(i_instrument = 0, instrument = driver->instruments;
	    i_instrument < driver->number_of_instruments;
	    i_instrument++, instrument++) {

		/* Sample擾܂B */
		instrument->number_of_samples_in_instrument = LEHALF(instrument_header->number_of_samples_in_instrument);

		/* ȉ́ASamplêInstrumentɑ΂Ă̂ݏs܂B
		 * ȂASamplêȂInstrument݂̂悭킩Ȃ(Ɏg?)̂łAdlłB
		 * ۂɁASamplêȂInstrument܂XMt@C݂͑AInstrumentŔ悤Ƃ܂B
		 * 2004/11/11݁ASamplêȂInstrument͖̔悤ɎĂ܂B
		 * ڂ́Axm_run()̎QƂĂB
		 */
		if(instrument->number_of_samples_in_instrument) {

			/* Sample headerTCY擾܂B
			 * ЂƂInstrumentɕSample܂܂ꍇA̎Sample header͓TCYłB
			 */
			sample_header_size = LEWORD(instrument_header->sample_header_size);

			/* Sample number for all notes擾܂B(Note0`95ɑΉSampleԍ) */
			instrument->sample_number_for_all_notes = instrument_header->sample_number_for_all_notes;
			for(note = 0; note < 96; note++) {
				i_sample = instrument->sample_number_for_all_notes[note];
				if((i_sample < 0) || (i_sample > instrument->number_of_samples_in_instrument - 1)) {
					return ERRNO;
				}
			}

			/* Volume type On = 1 Ȃ΁APoints for volume envelopeLłB */
			volume_type = LEBYTE(instrument_header->volume_type);
			if(volume_type & 1) {

				/* Point擾܂B
				 * ModPlugTrackerEnvelopeҏWŁAVolume envelopeOnɂ̏ԂłPoint2łB
				 * Point2̎ɁAPoint폜͂ł܂BlPoint12̎Pointǉ͂ł܂B
				 * K؂TrackerŕҏWXMt@Cł΁AsPoint͂肦Ȃ͂łB
				 * sPointXMt@C̓G[ƌȂAĐsƂĂ܂č\ȂƎv܂B
				 */
				number_of_volume_points = LEBYTE(instrument_header->number_of_volume_points);
				if((number_of_volume_points < 2) &&
				   (number_of_volume_points > 12)) {
					return ERRNO; /* Points */
				}

				/* Points for volume envelopeRs[邽߂XMENVELOPEPOINT\̔zmۂ܂B
				 * * Points for volume envelopeRs[闝R͎̂ƂłB
				 * - Points for volume envelopê܂܎gƂƁAevfHalfWordȂ̂ŁALEHALF()}NKvłB
				 *   tɕpɂLEHALF()}NgƁAxቺSzȂ̂ŁAVmۂzɃRs[邱Ƃɂ܂B
				 * - Ƃ͂A16bit sample datȁɂ́AtLEHALF()gĂ킯łAtLEHALF()gȂ킯ł͂ȂłB
				 *   16bit sample datȁɊrׂ΁AVolume envelopȅ̕ƏȂ̂ŁAxቺ̐SzKvȂm܂B
				 *   Axm_run()Volume envelope̎΂킩悤ɁAR[h̕GSampleVolume envelope̕łB
				 *   GȃR[h̒LEHALF()}N𑽗pƁAǐ邽߁A͂VmۂzɃRs[邱Ƃɂ܂B
				 */
				instrument->points_for_volume_envelope = calloc(number_of_volume_points, sizeof(XMENVELOPEPOINT));
				if(!instrument->points_for_volume_envelope) {
					return ERRNO;
				}
				memcpy(instrument       ->points_for_volume_envelope,
				       instrument_header->points_for_volume_envelope, sizeof(XMENVELOPEPOINT) * number_of_volume_points);

				/* * ePoint́AHalfWordl̑gŁA(x:Tick,y:Vol)̏Ŋi[Ă܂B
				 *   x͕KłBAׂ荇PointxlƂ͗L蓾܂B
				 * * XMt@C̎dlł́Ax:Tick=0`324Ay:Vol=0`64 ƂȂĂ܂B
				 *   AModPlugTrackerEnvelopeҏW@\́AƑ傫x:TickݒłĂ܂܂B
				 *   ܂Ay:Vol͈͊OłȂł̂ŁAȂƂɂ܂B(Ǝf)
				 */
				for(i_point = 0; i_point < number_of_volume_points; i_point++) {
					if(!i_point) {
						if(instrument->points_for_volume_envelope[i_point].x) {
							return ERRNO; /* ŏPointx:Tick0łȂ΂܂ */
						}
					} else {
						if(instrument->points_for_volume_envelope[i_point    ].x <
						   instrument->points_for_volume_envelope[i_point - 1].x) {
							return ERRNO; /* x:Tick̕т~ł(l͍\܂) */
						}
					}
				}

				/* * Volume type Sustain = 1 Ȃ΁AVolume sustain pointLłB
				 *   Volume type Sustain = 0 Ȃ΁AŌpointVolume sustain pointݒ肳Ă̂ƓłB
				 */
				if(volume_type & 2) {
					volume_sustain_point = LEBYTE(instrument_header->volume_sustain_point);
					if((volume_sustain_point < 0) ||
					   (volume_sustain_point > number_of_volume_points - 1)) {
						return ERRNO; /* Volume sustain points */
					}
				} else {
					volume_sustain_point = number_of_volume_points - 1;
				}
				instrument->volume_sustain_point = &instrument->points_for_volume_envelope[volume_sustain_point];

				/* * Volume type Loop = 1 Ȃ΁AVolume loop start/end pointLłB
				 *   Volume type Loop = 0 Ȃ΁AŌpointVolume loop start/end pointݒ肳Ă̂ƓłB
				 * * Volume loop end pointVolume loop start pointォA܂͓pointwĂȂ΂܂B
				 *   Volume loop end point == Volume loop start pointL聚ł!!
				 */
				if(volume_type & 4) {
					volume_loop_start_point = LEBYTE(instrument_header->volume_loop_start_point);
					volume_loop_end_point = LEBYTE(instrument_header->volume_loop_end_point);
					if((volume_loop_start_point < 0) ||
					   (volume_loop_start_point > number_of_volume_points - 1)) {
						return ERRNO; /* Volume loop start points */
					}
					if((volume_loop_end_point < 0) ||
					   (volume_loop_end_point > number_of_volume_points - 1)) {
						return ERRNO; /* Volume loop end points */
					}
					if(volume_loop_start_point > volume_loop_end_point) {
						return ERRNO; /* Volume loop start/end points */
					}
				} else {
					volume_loop_start_point = number_of_volume_points - 1;
					volume_loop_end_point = number_of_volume_points - 1;
				}
				instrument->volume_loop_start_point = &instrument->points_for_volume_envelope[volume_loop_start_point];
				instrument->volume_loop_end_point = &instrument->points_for_volume_envelope[volume_loop_end_point];
			}

			/* Volume fadeout擾܂B */
			instrument->volume_fadeout = LEHALF(instrument_header->volume_fadeout);

			/* ЂƂ߂SampleSample headerAhXƁAЂƂ߂SampleSample dataAhX擾܂B
			 * Instrument̍\́A
			 *   |	Instrument header
			 *   |	Sample header #0
			 *   |	Sample header #1
			 *   |	      .
			 *   |	      .
			 *   |	Sample header #(Number of samples in instrument - 1)
			 *   |	Sample data   #0
			 *   |	Sample data   #1
			 *   |	      .
			 *   |	      .
			 *   |	Sample data   #(Number of samples in instrument - 1)
			 * łB
			 *   |	Instrument header
			 *   |	Sample header #0
			 *   |	Sample data   #0
			 *   |	Sample header #1
			 *   |	Sample data   #1
			 *   |	      .
			 *   |	      .
			 *   |	Sample header #(Number of samples in instrument - 1)
			 *   |	Sample data   #(Number of samples in instrument - 1)
			 * ł͂܂!!ӂĂB
			 */
			sample_header = (const XMSAMPLEHEADER*)((int)instrument_header +
							      LEWORD(instrument_header->instrument_size));
			sample_data = (const unsigned char*)((int)sample_header +
								  sample_header_size * instrument->number_of_samples_in_instrument);

			/* XMSAMPLE\̂mۂ܂B */
			instrument->samples = calloc(instrument->number_of_samples_in_instrument, sizeof(XMSAMPLE));
			if(!instrument->samples) {
				return ERRNO;
			}

			/* eSampleɂ... */
			for(i_sample = 0, sample = instrument->samples;
			    i_sample < instrument->number_of_samples_in_instrument;
			    i_sample++, sample++) {

				/* etB[hl擾܂B */
				sample->sample_data = sample_data;
				sample_length = LEWORD(sample_header->sample_length);
				sample_loop_start = LEWORD(sample_header->sample_loop_start);
				sample_loop_length = LEWORD(sample_header->sample_loop_length);
				sample->volume = LEBYTE(sample_header->volume);
				sample->finetune = (char)LEBYTE(sample_header->finetune);
				sample->type = LEBYTE(sample_header->type);
				sample->relative_note_number = (char)LEBYTE(sample_header->relative_note_number);

				///* 0Sample̓G[ƌȂƂɂ܂B(Ǝf) */
				//if(sample_length < 1) {
				//	return ERRNO;
				//}
				//SampleɃRg𖄂ߍނ߂̃_~[ƂāA0SampleL蓾悤łB
				//0SampleXMt@CȂ肠悤Ȃ̂ŁAG[ȂƂɂ܂B
				//0SamplegĔ邱Ƃ͖Ǝv܂Aꔭꍇɒ~悤A
				//mNo loopƂĂƂɂ܂B(Ǝf)
				if(!sample_length) {
					sample->type &= ~3; /* No loop */
				}

				/* [v0Ȃ΁AForward loopPing-pong loop𖳌܂B(dl) */
				if(!sample_loop_length) {
					sample->type &= ~3; /* No loop */
				}

				/* SampleA[vJnʒuA[v́AoCgPʂłB
				 * 16bit SamplȅꍇA̒l͕KɂȂĂ͂łB
				 * Ȃ΁AG[ƌȂƂɂ܂B(Ǝf)
				 */
				if(sample->type & 0x10) { /* 16bit */
					if(sample_length & 1) {
						return ERRNO;
					}
					if(sample->type & 3) { /* Forward loop, Ping-pong loop */
						if(sample_loop_start & 1) {
							return ERRNO;
						}
						if(sample_loop_length & 1) {
							return ERRNO;
						}
					}
				}

				if(!(sample->type & 3)) { /* No loop */

					/* SampleI[ʒui[܂B */
					sample->sample_end = sample->sample_data +
								     sample_length;

				} else { /* Forward loop, Ping-pong loop */

					/* [vJnʒuƃ[v܂B */
					if((sample_loop_start < 0) ||
					   (sample_loop_start > sample_length - 1)) {
						return ERRNO;
					}
					if(sample_loop_start + sample_loop_length > sample_length) {
						return ERRNO;
					}

					/* [vJnʒuƃ[vI[ʒui[܂B */
					sample->sample_loop_start = sample->sample_data +
									    sample_loop_start;
					sample->sample_loop_end = sample->sample_loop_start +
									  sample_loop_length;
				}

				/* Forward loop̏ꍇ̂݁A[vJnʒũTvo͂vZĂ܂B */
				switch(sample->type & 0x13) {
				case 0x00: /* No loop, 8bit */
				case 0x10: /* No loop, 16bit */
					/** no job **/
					break;
				case 0x01: /* Forward loop, 8bit */
					for(sample_position = sample->sample_data;
					    sample_position < sample->sample_loop_start;
					    sample_position += 1) {
						sample->sample_loop_output += LEBYTE(sample_position) << 8; /* D31:16͕s */
					}
					break;
				case 0x11: /* Forward loop, 16bit */
					for(sample_position = sample->sample_data;
					    sample_position < sample->sample_loop_start;
					    sample_position += 2) {
						sample->sample_loop_output += LEHALF(sample_position); /* D31:16͕s */
					}
					break;
				case 0x02: /* Ping-pong loop, 8bit */
				case 0x12: /* Ping-pong loop, 16bit */
					/** no job **/
					break;
				default:
					return ERRNO; /* Types */
				}

				/* Sample֐i߂܂B */
				sample_header = (const XMSAMPLEHEADER*)((int)sample_header +
									     sample_header_size);
				sample_data = (const unsigned char*)((int)sample_data +
									  sample_length);
			}
	
			/* Instrument֐i߂܂B */
			instrument_header = (const XMINSTRUMENTHEADER*)sample_data;

		} else {

			/* Instrument֐i߂܂B */
			instrument_header = (const XMINSTRUMENTHEADER*)((int)instrument_header +
								      LEWORD(instrument_header->instrument_size));
		}
	}

	/************************
	 *  tJn̏ݒ  *
	 ************************/

	/* Instrumentŏ̔vꂽꍇɔāAInstrument#0ݒ肵Ă܂B(Ǝf) */
	for(i_channel = 0, channel = driver->channels;
	    i_channel < driver->number_of_channels;
	    i_channel++, channel++) {
		channel->pending_instrument = &driver->instruments[0];
	}

	/* ŏPattern擾܂B */
	i_pattern = driver->pattern_order_table[driver->order];
	if((i_pattern < 0) ||
	   (i_pattern > driver->number_of_patterns - 1)) {
		return ERRNO; /* Patterns */
	}
	driver->pattern = &driver->patterns[i_pattern];

	/* Packed pattern data̐擪AhX擾܂B */
	driver->packed_pattern_position = driver->pattern->packed_pattern_data;

	/* Global volume܂B */
	driver->global_volume = 64;

	/* Position jumpAPattern breakNA܂B */
	driver->position_jump = -1;
	driver->pattern_break = -1;

	return 0;
}

/* XM Driver܂B
 * [in]
 *	driver		XMDRIVER\́B
 *	data		XMf[^̐擪AhXB
 * [out]
 *	߂l		Ȃ0Ԃ܂B
 *			sȂ0ȊO̒lԂ܂B
 * [note]
 *	* xm_init()ꍇ́AAxm_init()ĂяoOɕKxm_free()ĂяoĂB
 *	  xm_free()Ăяoxm_init()ȏĂяoƁA[N܂B
 *	* xm_init()sꍇ́AɃ͊JĂ̂ŁAxm_free()ĂяoKv͂܂B
 *	  xm_init()sꍇAxm_free()ĂяoĂ\܂B
 *	  xm_free()̌ĂяóAtāAɈSłB
 */
int
xm_init(XMDRIVER* driver, const void* data)
{
	int retval;

	retval = _xm_init(driver, data);
	if(retval != 0) {
		xm_free(driver); /* J */
	}

	return retval;
}

/* xm_init()ɂĊmۂJ܂B
 * [in]
 *	driver		XMDRIVER\́B
 * [note]
 *	* xm_init()ĂяoɁAxm_free()ȏĂяoĂ\܂B
 */
void
xm_free(XMDRIVER* driver)
{
	int i_instrument;
	XMINSTRUMENT* instrument;

	/* XMPATTERN\̂J܂B */
	if(driver->patterns) {
		free(driver->patterns);
	}

	/* XMINSTRUMENT\́AXMSAMPLE\́AXMENVELOPEPOINT\̔zJ܂B */
	if(driver->instruments) {
		for(i_instrument = 0, instrument = driver->instruments;
		    i_instrument < driver->number_of_instruments;
		    i_instrument++, instrument++) {
			if(instrument->samples) {
				free(instrument->samples);
			}
			if(instrument->points_for_volume_envelope) {
				free(instrument->points_for_volume_envelope);
			}
		}
		free(driver->instruments);
	}

	/* XMCHANNEL\̂J܂B */
	if(driver->channels) {
		free(driver->channels);
	}

	/* xm_free()ȏĂ΂ꂽꍇ̂߂ɁAXMDRIVER\̂NAĂ܂B */
	memset(driver, 0, sizeof(XMDRIVER));
}

int
xm_stream_callback(short* wbuff, int param)
{
	XMDRIVER* driver = (XMDRIVER*)param;
	//
	int retval;

	retval = xm_run(driver);
	if(retval == 0) {
		retval = xm_mix(driver, wbuff);
	}

	return retval;
}

/****************************************************************************
 *	AvP[Vp֐
 ****************************************************************************/

static XMDRIVER driver;

int
xm_play(const void* data)
{
	int retval;

	/* ܂mɒ~܂B */
	xm_stop();

	/* XM Driver܂B */
	retval = xm_init(&driver, data);
	if(retval != 0) {
		return retval;
	}

	/* Xg[ĐJn܂B */
	stream_play(XMBUFLEN, xm_stream_callback, (int)&driver, 1/*X^bN؊gp*/);

	return 0;
}

void
xm_stop()
{
	/* mɃXg[Đ~܂B */
	stream_stop();

	/* mXM DriverJ܂B */
	xm_free(&driver);
}

