/*	
 *	clips26.c
 *
 *	P/ECE Signetics 2650 Emulator
 *
 *	CLiP - Common Library for P/ECE
 *	Copyright (C) 2001-2005 Naoyuki Sawa
 *
 *	* Mon Apr 25 20:42:00 JST 2005 Naoyuki Sawa
 *	- 쐬JnB
 */
#include "clip.h"

/* S2650_TRACEV{`ƁAg[Xo͂s܂B */
//#define S2650_TRACE

/****************************************************************************
 *	O[oϐ
 ****************************************************************************/

#ifndef PIECE /*============================================================*/
S2650 s2650; /* P/ECȄꍇ͍RAMɔzu -> frams26a.s */
#endif /*PIECE==============================================================*/

#include "s2650/decl.h"
const S2650SEQ s2650seq[0x100] = { /* V[PXe[u */
#include "s2650/seq.h"
};

/****************************************************************************
 *	[Jϐ
 ****************************************************************************/

#define cpu	(&s2650)

#define PSU_SP0	(1<<0)
#define PSU_SP1	(1<<1)
#define PSU_SP2	(1<<2)
// Unused	(1<<3)
// Unused	(1<<4)
#define PSU_II	(1<<5)
#define PSU_F	(1<<6)
#define PSU_S	(1<<7)
#define PSU_SP	(PSU_SP2|PSU_SP1|PSU_SP0)

#define PSL_C	(1<<0)
#define PSL_COM	(1<<1)
#define PSL_OVF	(1<<2)
#define PSL_WC	(1<<3)
#define PSL_RS	(1<<4)
#define PSL_IDC	(1<<5)
#define PSL_CC0	(1<<6)
#define PSL_CC1	(1<<7)
#define PSL_CC	(PSL_CC1|PSL_CC0)

#ifndef PIECE /*============================================================*/
static void (* const* seq)();
static unsigned char* reg;
static int addr;
static int data;
static int cond;
#ifdef S2650_TRACE
static const char* const s2650mne[0x100] = { /* j[jbNe[u */
#include "s2650/mne.h"
};
#endif /*S2650_TRACE*/
#endif /*PIECE==============================================================*/

/****************************************************************************
 *	֐
 ****************************************************************************/

void
s2650_err()
{
	die("s2650_error(%04x)", cpu->iar);
}

#ifndef PIECE /*============================================================*/

static void
s2650_check_pending()
{
	/* 荞ݗvZbgĂȂ΁A܂B */
	if(!(cpu->flags & S2650_IRQ)) {
		return;
	}

	/* 荞݋֎~(II=1)Ȃ΁A܂B */
	if(cpu->psu & PSU_II) {
		return;
	}

	/* 荞ݗvNABHALTȂHALT܂B */
	cpu->flags &= ~(S2650_IRQ | S2650_HALT);

	/* 荞݋֎~(II=1)܂B */
	cpu->psu |= PSU_II;

	/* "ZBSR vector"̏s܂B */
	{
		static void s2650_VEC(); /* OQ */
		static void s2650_BS();  /* OQ */
		data = cpu->vector;
		s2650_VEC(); /* AddressMode */
		s2650_BS();
	}
}

#endif /*PIECE==============================================================*/

/****************************************************************************
 *	AvP[Vp֐
 ****************************************************************************/

int
s2650_get_psu()
{
	int psu;

	psu = cpu->psu & (PSU_S | PSU_F | PSU_II);	/* S|F|II|0|0|  0|  0|  0 */
	psu |= cpu->sp;					/* S|F|II|0|0|SP2|SP1|SP0 */

	return psu;
}

void
s2650_set_psu(int psu)
{
	cpu->psu = (unsigned char)((cpu->psu & PSU_S) | (psu & ~PSU_S));
	cpu->sp  = (unsigned char)(psu & PSU_SP);
}

int
s2650_get_psl()
{
	int psl;

	psl = cpu->psl & (PSL_RS | PSL_WC | PSL_COM);	/*   0|  0|  0|RS|WC|  0|COM|0 */
	if(cpu->c & 0x100) {				/*   0|  0|  0|RS|WC|  0|COM|C */
		psl |= PSL_C;
	}
	if(cpu->ovf & 0x80) {				/*   0|  0|  0|RS|WC|OVF|COM|C */
		psl |= PSL_OVF;
	}
	if(cpu->idc & 0x10) {				/*   0|  0|IDC|RS|WC|OVF|COM|C */
		psl |= PSL_IDC;
	}
	if(cpu->cc) {					/* CC1|CC0|IDC|RS|WC|OVF|COM|C */
		if(cpu->cc < 0x80) {
			psl |= PSL_CC0;
		} else if(cpu->cc < 0x100) {
			psl |= PSL_CC1;
		} else {
			psl |= PSL_CC1 | PSL_CC0;
		}
	}

	return psl;
}

void
s2650_set_psl(int psl)
{
	cpu->psl = psl;
	cpu->c   = psl << (8-0);
	cpu->ovf = psl << (7-2);
	cpu->idc = psl >> (5-4);
	cpu->cc  = psl >> (6-0);	/* 0_00000000 or 0_00000001 or 0_00000010 or 0_00000011 */
	cpu->cc <<= cpu->cc;		/* 0_00000000 or 0_00000010 or 0_00001000 or 0_00011000 */
	cpu->cc <<= 4;			/* 0_00000000 or 0_00100000 or 0_10000000 or 1_10000000 */
}

void
s2650_set_sense(int sense)
{
	if(sense) {
		cpu->psu |=  PSU_S;
	} else {
		cpu->psu &= ~PSU_S;
	}
}

void
s2650_reset()
{
	/* RESET͈̓ȉ̒ʂłB[2650}jA(p.29)]
	 * EIAR     = 0x0000
	 * EPSU(II) = 0:荞݋ (Mǎdlł)
	 * Ȇ͕s (?)
	 * ]āA\̃NAOKłB
	 */
	memset(cpu, 0, sizeof(S2650));
}

#ifndef PIECE /*============================================================*/

void
s2650_irq(int vector)
{
	/* 荞ݗvZbg܂B */
	cpu->flags |= S2650_IRQ;

	/* 荞݃xN^i[܂B */
	cpu->vector = vector;

	/* Ɋ荞ݎs\ׂ܂B */
	s2650_check_pending();
}

void
s2650_run(int cycle)
{
	int opcode;
	const S2650SEQ* seqhdr;

	/* cTCNi[܂B */
	cpu->cycle = cycle;

	/* HALTȂ΁AcsTCN0ɂāAɃ[v𔲂܂B */
	if(cpu->flags & S2650_HALT) {
		cpu->cycle = 0; /* (s2650.cycle=0Ŋ֐𔲂邱ƁBK{!!) */
	}

	/* csTCN0ȉɂȂ܂ŌJԂ܂B */
	while(cpu->cycle > 0) {

		/* OpCodetFb`܂B */
		opcode = s2650_read(cpu->iar++);
#ifdef S2650_TRACE
		{
			int psu = s2650_get_psu();
			int psl = s2650_get_psl();
			printf("%04x: %-16s %02x %02x %02x %02x %02x %02x %02x %c%c%c%c%c%c%c%c %c%c%c%c%c%c%c%c\n",
				cpu->iar-1,s2650mne[opcode],
				cpu->reg[0],cpu->reg[1],cpu->reg[2],cpu->reg[3],cpu->reg[4],cpu->reg[5],cpu->reg[6],
				psu&0x80?'S':'-',psu&0x40?'F':'-',psu&0x20?'I':'-',psu&0x10?'?':'-',
				psu&0x08?'?':'-',psu&0x04?'1':'0',psu&0x02?'1':'0',psu&0x01?'1':'0',
				psl&0x80?'1':'0',psl&0x40?'1':'0',psl&0x20?'D':'-',psl&0x10?'R':'-',
				psl&0x08?'W':'-',psl&0x04?'O':'-',psl&0x02?'M':'-',psl&0x01?'C':'-');
		}
#endif /*S2650_TRACE*/

		/* OpCodeɑΉV[PX̐擪AhX擾܂B */
		seqhdr = &s2650seq[opcode];

		/* csTCN炵܂B(ߎsOɌ炷!!) */
		cpu->cycle -= seqhdr->cycle;

		/* V[PX́Aŏ̊֐Ăт܂B */
#ifdef DEBUG
		/* V[PX̌ĂіYꌟô߁AlŒ肵Ă܂B */
		reg = NULL;
		addr = -1;
		data = -1;
		cond = -1;
#endif /*DEBUG*/
		seq = &seqhdr->fn1;
		(*seq++)();
	}
}

/****************************************************************************
 *	Register
 ****************************************************************************/

void
s2650_R0()
{
	reg = &cpu->reg[0];
}
void
s2650_R1()
{
	reg = &cpu->reg[1];
	if(cpu->psl & PSL_RS) {
		reg += 3;
	}
}
void
s2650_R2()
{
	reg = &cpu->reg[2];
	if(cpu->psl & PSL_RS) {
		reg += 3;
	}
}
void
s2650_R3()
{
	reg = &cpu->reg[3];
	if(cpu->psl & PSL_RS) {
		reg += 3;
	}
}

/****************************************************************************
 *	Condition
 ****************************************************************************/

void
s2650_EQ()
{
	/* if cc=0 (CC=00) then cond=0 */
	cond = cpu->cc;
}
void
s2650_GT()
{
	/* if cc=0x01..0x7f (CC=01) then cond=0 */
	cond = (cpu->cc < 0x01) || (cpu->cc > 0x7f);
}
void
s2650_LT()
{
	/* if cc=0x80..0xff (CC=10) then cond=0 */
	cond = (cpu->cc < 0x80) || (cpu->cc > 0xff);
}
void
s2650_UN()
{
	/* always cond=0 */
	cond = 0;
}

void
s2650_IR()
{
	(*seq++)(); /* Register */
	(*reg)++;
	cond = *reg;
}
void
s2650_DR()
{
	(*seq++)(); /* Register */
	(*reg)--;
	cond = *reg;
}
void
s2650_NR()
{
	(*seq++)(); /* Register */
	cond = *reg;
}

/****************************************************************************
 *	AddressMode
 ****************************************************************************/

static void
s2650_IND() /* INDirect addressing */
{
	/* ̃AhX[h֐Ă΂܂Bdataj󂵂Ȃ悤!! */
	int tmp;
	tmp  = s2650_read(addr++) << 8;		/* tmp  = ?ppaaaaa_00000000 */
	tmp |= s2650_read(addr  );		/* tmp  = ?ppaaaaa_aaaaaaaa */
	addr = tmp & 0x7fff;			/* addr = 0ppaaaaa_aaaaaaaa */
	cpu->cycle -= 6; /* C_CNgAhX[h̏TCN */
}
static void
s2650_ABB() /* ABsolute addressing for Branch instructions */
{
	data  = s2650_read(cpu->iar++) << 8;	/* data = ippaaaaa_00000000 */
	data |= s2650_read(cpu->iar++);		/* data = ippaaaaa_aaaaaaaa */
	addr  = data & 0x7fff;			/* data = 0ppaaaaa_aaaaaaaa */
	if(data & 0x8000) { /* C_CNgAhX[h? */
		s2650_IND();
	}
}
static void
s2650_AXB() /* Absolute indeXed addressing for Branch instructions */
{
	s2650_ABB(); /* AddressMode */
	s2650_R3();  /* Register */
	addr = ( addr         & 0x6000) |
	       ((addr + *reg) & 0x1fff);
}
static void
s2650_VEC() /* Relative addressing from Zero Page [in] data=vector */
{
	int rel;
	/* [in] data=vector */			/* data = irrrrrrr */
	rel  = (char)(data << 1) >> 1;		/* rel  = -64..+63 */
	addr = rel & 0x1fff;
	if(data & 0x80) { /* C_CNgAhX[h? */
		s2650_IND();
	}
}
static void
s2650_ZPR() /* Relative addressing from Zero Page (vectortFb`) */
{
	data = s2650_read(cpu->iar++);		/* data = irrrrrrr */
	s2650_VEC();
}

void
s2650_IMM() /* IMMediate addressing */
{
	addr = cpu->iar++;
}
void
s2650_REL() /* RELative addressing */
{
	int rel;
	data = s2650_read(cpu->iar++);		/* data = irrrrrrr */
	rel  = (char)(data << 1) >> 1;		/* rel  = -64..+63 */
	addr = ( cpu->iar        & 0x6000) |
	       ((cpu->iar + rel) & 0x1fff);
	if(data & 0x80) { /* C_CNgAhX[h? */
		s2650_IND();
	}
}
void
s2650_ABS() /* ABSolute addressing for non-branch instructions */
{
	data  = s2650_read(cpu->iar++) << 8;	/* data = ixxaaaaa_00000000 */
	data |= s2650_read(cpu->iar++);		/* data = ixxaaaaa_aaaaaaaa */
	addr = (cpu->iar & 0x6000) |
	       (data     & 0x1fff);
	if(data & 0x8000) { /* C_CNgAhX[h? */
		s2650_IND();
	}
	if(data & 0x6000) { /* CfNXAhX[h? */
		/* * D14܂D131Ȃ΁ACfNXAhX[hłB
		 *   OpCodẽWX^ẃACfNXWX^܂B
		 * - D13=1Ȃ΁AɃCfNXWX^{PǍAhXɉZ܂B
		 * - D14=1Ȃ΁AɃCfNXWX^{PǍAhXɉZ܂B
		 * - D13=1D14=1Ȃ΁ACfNXWX^̂܂܃AhXɉZ܂B
		 *   ȉ̏ł́}PEAʓIɂ̂܂܃AhXɉZĂ܂B
		 * * CfNXAhX[hł́A߂̑ΏۃWX^͈ÖقR0ƂȂ܂B
		 */
		if(data & 0x2000) { /* D14:13=01or11 */
			(*reg)++;
		}
		if(data & 0x4000) { /* D14:13=10or11 */
			(*reg)--;
		}
		addr = ( addr         & 0x6000) |
		       ((addr + *reg) & 0x1fff);
		s2650_R0(); /* Register */
	}
}

void
s2650_CTL() /* M/#IO=#IO, D/#C=#C(Control), E/#NE=#NE(Non-extended) */
{
	addr = 0x100; /* {̓AhXoXɂ͉o͂܂ */
}
void
s2650_DAT() /* M/#IO=#IO, D/#C=D(Data), E/#NE=#NE(Non-extended) */
{
	addr = 0x101; /* {̓AhXoXɂ͉o͂܂ */
}
void
s2650_EXT() /* M/#IO=#IO, E/#NE=E(Extended), A7:0=2nd byte of the instruction */
{
	addr = s2650_read(cpu->iar++);
}

/****************************************************************************
 *	Operation
 ****************************************************************************/

void
s2650_ERR()
{
	s2650_err();
}

/*--------------------------------------------------------------------------*/

void
s2650_LODr()
{
	(*seq++)(); /* Register */
	cpu->reg[0] = *reg;
	cpu->cc = cpu->reg[0];
}
void
s2650_LODm()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	*reg = s2650_read(addr);
	cpu->cc = *reg;
}

void
s2650_STRr()
{
	(*seq++)(); /* Register */
	*reg = cpu->reg[0];
	cpu->cc = *reg;
}
void
s2650_STRm()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	s2650_write(addr, *reg);
	/* CC not affected */
}

/*--------------------------------------------------------------------------*/

static void
s2650_ADD(int a, int b)
{
	int c;
	int t;

	c = 0;
	if(cpu->psl & PSL_WC) {
		c = (cpu->c >> 8) & 1;
	}
	t = a + b + c;

	cpu->c   = t;
	cpu->ovf = ~(a ^ b) & (a ^ t); /* vmF */
	cpu->idc = (a & 0xf) + (b & 0xf) + c;

	cpu->cc = (unsigned char)t;
}
void
s2650_ADDr()
{
	(*seq++)(); /* Register */
	s2650_ADD(cpu->reg[0], *reg);
	cpu->reg[0] = (unsigned char)cpu->cc;
}
void
s2650_ADDm()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	s2650_ADD(*reg, s2650_read(addr));
	*reg = (unsigned char)cpu->cc;
}

static void
s2650_SUB(int a, int b)
{
	int c;
	int t;

	/* Signetics 2650CtOdĺA6502ƓlŁAZ0/1tłB
	 * IDCtO(n[tL[tO)AZ0/1tƂȂ܂B
	 */
	c = 0;
	if(cpu->psl & PSL_WC) {
		c = (~cpu->c >> 8) & 1;
	}
	t = a - b - c;

	cpu->c   = ~t;
	cpu->ovf = (a ^ b) & (a ^ t); /* vmF */
	cpu->idc = ~((a & 0xf) - (b & 0xf) - c);

	cpu->cc = (unsigned char)t;
}
void
s2650_SUBr()
{
	(*seq++)(); /* Register */
	s2650_SUB(cpu->reg[0], *reg);
	cpu->reg[0] = (unsigned char)cpu->cc;
}
void
s2650_SUBm()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	s2650_SUB(*reg, s2650_read(addr));
	*reg = (unsigned char)cpu->cc;
}

void
s2650_ANDr()
{
	(*seq++)(); /* Register */
	cpu->cc = cpu->reg[0] & *reg;
	cpu->reg[0] = (unsigned char)cpu->cc;
}
void
s2650_ANDm()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	cpu->cc = *reg & s2650_read(addr);
	*reg = (unsigned char)cpu->cc;
}

void
s2650_IORr()
{
	(*seq++)(); /* Register */
	cpu->cc = cpu->reg[0] | *reg;
	cpu->reg[0] = (unsigned char)cpu->cc;
}
void
s2650_IORm()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	cpu->cc = *reg | s2650_read(addr);
	*reg = (unsigned char)cpu->cc;
}

void
s2650_EORr()
{
	(*seq++)(); /* Register */
	cpu->cc = cpu->reg[0] ^ *reg;
	cpu->reg[0] = (unsigned char)cpu->cc;
}
void
s2650_EORm()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	cpu->cc = *reg ^ s2650_read(addr);
	*reg = (unsigned char)cpu->cc;
}

static void
s2650_COM(int a, int b)
{
	if(!(cpu->psl & PSL_COM)) { /* COM=0:Arithmetic/1:Logical */
		a = (char)a;
		b = (char)b;
	}
	if(a == b) {
		cpu->cc = 0x00;
	} else if(a > b) {
		cpu->cc = 0x01;
	} else /*(a < b)*/ {
		cpu->cc = 0x80;
	}
}
void
s2650_COMr()
{
	(*seq++)(); /* Register */
	s2650_COM(cpu->reg[0], *reg);
}
void
s2650_COMm()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	s2650_COM(*reg, s2650_read(addr));
}

void
s2650_RRL()
{
	(*seq++)(); /* Register */
	data = *reg;
	if(cpu->psl & PSL_WC) {
		int c    = cpu->c >> 8;			/*        ???????c */
		data     = (data << 1) | (c & 1);	/*      d_dddddddc */
		cpu->c   = data;			/*   C<-+   |      */
		cpu->idc = data >> (5-4);		/* IDC<-----+      */
	} else {
		data = (data << 1) | (data >> 7);
	}
	cpu->ovf = data ^ *reg; /* vmF */
	*reg = data;
	cpu->cc = *reg;
}
void
s2650_RRR()
{
	(*seq++)(); /* Register */
	data = *reg;
	if(cpu->psl & PSL_WC) {
		int c    = cpu->c >> 8;			/*     ???????c    */
		cpu->c   = data << 8;			/*     dddddddd->C */
		data     = (data >> 1) | (c << 7);	/*   ?_cddddddd    */
		cpu->idc = data >> (5-4);		/* IDC<--+         */
	} else {
		data = (data >> 1) | (data << 7);
	}
	cpu->ovf = data ^ *reg; /* vmF */
	*reg = data;
	cpu->cc = *reg;
}

/*--------------------------------------------------------------------------*/

void
s2650_LPSU()
{
	s2650_set_psu(cpu->reg[0]);
	s2650_check_pending(); /* II=1->0̉\̂ */
}
void
s2650_LPSL()
{
	s2650_set_psl(cpu->reg[0]);
}

void
s2650_SPSU()
{
	cpu->reg[0] = s2650_get_psu();
	cpu->cc = cpu->reg[0];
}
void
s2650_SPSL()
{
	cpu->reg[0] = s2650_get_psl();
	cpu->cc = cpu->reg[0];
}

void
s2650_PPSU()
{
	int psu = s2650_get_psu();
	(*seq++)(); /* AddressMode */
	data = s2650_read(addr);
	psu |= data;
	s2650_set_psu(psu);
}
void
s2650_PPSL()
{
	int psl = s2650_get_psl();
	(*seq++)(); /* AddressMode */
	data = s2650_read(addr);
	psl |= data;
	s2650_set_psl(psl);
}

void
s2650_CPSU()
{
	int psu = s2650_get_psu();
	(*seq++)(); /* AddressMode */
	data = s2650_read(addr);
	psu &= ~data;
	s2650_set_psu(psu);
	s2650_check_pending(); /* II=1->0̉\̂ */
}
void
s2650_CPSL()
{
	int psl = s2650_get_psl();
	(*seq++)(); /* AddressMode */
	data = s2650_read(addr);
	psl &= ~data;
	s2650_set_psl(psl);
}

static void
s2650_TEST(int a, int b)
{
	/* b̒1̃rbg     a̒łS1Ȃ   cpu->cc=0x0000 (CC=00)
	 * b̒1̃rbĝ a̒ňł0Ȃ cpu->cc=0x00ff (CC=10)
	 * -------------------------- ؃vO --------------------------
	 * int main() {
	 * 	int a, b, cc;
	 * 	for(a = 0x00; a <= 0xff; a++) {
	 * 		for(b = 0x00; b <= 0xff; b++) {
	 * 			cc = (unsigned char)(((a & b) - b) >> 8);
	 * 			if((a & b) == b) {
	 * 				if(cc != 0x00) {
	 * 					printf("Miss!\n");
	 * 					exit(-1);
	 * 				}
	 * 			} else {
	 * 				if(cc != 0xff) {
	 * 					printf("Miss!\n");
	 * 					exit(-1);
	 * 				}
	 * 			}
	 * 		}
	 * 	}
	 * 	printf("OK\n");
	 * 	return 0;
	 * }
	 */
	cpu->cc = (unsigned char)(((a & b) - b) >> 8);
}
void
s2650_TPSU()
{
	(*seq++)(); /* AddressMode */
	s2650_TEST(s2650_get_psu(), s2650_read(addr));
}
void
s2650_TPSL()
{
	(*seq++)(); /* AddressMode */
	s2650_TEST(s2650_get_psl(), s2650_read(addr));
}

/*--------------------------------------------------------------------------*/

static void
s2650_BRR()
{
	s2650_REL(); /* AddressMode */
	cpu->iar = addr;
}
static void
s2650_BRA()
{
	s2650_ABB(); /* AddressMode */
	cpu->iar = addr;
}

void
s2650_BRReq()
{
	(*seq++)(); /* Condition */
	if(!cond) {
		s2650_BRR();
	} else {
		cpu->iar += 1;
	}
}
void
s2650_BRAeq()
{
	(*seq++)(); /* Condition */
	if(!cond) {
		s2650_BRA();
	} else {
		cpu->iar += 2;
	}
}

void
s2650_BRRne()
{
	(*seq++)(); /* Condition */
	if(cond) {
		s2650_BRR();
	} else {
		cpu->iar += 1;
	}
}
void
s2650_BRAne()
{
	(*seq++)(); /* Condition */
	if(cond) {
		s2650_BRA();
	} else {
		cpu->iar += 2;
	}
}

void
s2650_ZBRR()
{
	s2650_ZPR(); /* AddressMode */
	cpu->iar = addr;
}
void
s2650_BXA()
{
	s2650_AXB(); /* AddressMode */
	cpu->iar = addr;
}

/*--------------------------------------------------------------------------*/

static void
s2650_BS()
{
	cpu->sp = (cpu->sp + 1) & 7;
	cpu->ras[cpu->sp] = cpu->iar;
	cpu->iar = addr;
}
static void
s2650_BSR()
{
	s2650_REL(); /* AddressMode */
	s2650_BS();
}
static void
s2650_BSA()
{
	s2650_ABB(); /* AddressMode */
	s2650_BS();
}

void
s2650_BSReq()
{
	(*seq++)(); /* Condition */
	if(!cond) {
		s2650_BSR();
	} else {
		cpu->iar += 1;
	}
}
void
s2650_BSAeq()
{
	(*seq++)(); /* Condition */
	if(!cond) {
		s2650_BSA();
	} else {
		cpu->iar += 2;
	}
}

void
s2650_BSRne()
{
	(*seq++)(); /* Condition */
	if(cond) {
		s2650_BSR();
	} else {
		cpu->iar += 1;
	}
}
void
s2650_BSAne()
{
	(*seq++)(); /* Condition */
	if(cond) {
		s2650_BSA();
	} else {
		cpu->iar += 2;
	}
}

void
s2650_ZBSR()
{
	s2650_ZPR(); /* AddressMode */
	s2650_BS();
}
void
s2650_BSXA()
{
	s2650_AXB(); /* AddressMode */
	s2650_BS();
}

/*--------------------------------------------------------------------------*/

static void
s2650_RET()
{
	cpu->iar = cpu->ras[cpu->sp];
	cpu->sp = (cpu->sp - 1) & 7;
}
void
s2650_RETC()
{
	(*seq++)(); /* Condition */
	if(!cond) {
		s2650_RET();
	}
}
void
s2650_RETE()
{
	(*seq++)(); /* Condition */
	if(!cond) {
		s2650_RET();
		cpu->psu &= ~PSU_II;
		s2650_check_pending(); /* II=1->0̉\̂ */
	}
}

/*--------------------------------------------------------------------------*/

void
s2650_REDp()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	*reg = s2650_read_io(addr);
	cpu->cc = *reg;
}

void
s2650_WRTp()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	s2650_write_io(addr, *reg);
}

/*--------------------------------------------------------------------------*/

void
s2650_NOP()
{
	/** no job **/
}

void
s2650_TMI()
{
	(*seq++)(); /* Register */
	(*seq++)(); /* AddressMode */
	data = s2650_read(addr);
	if((*reg & data) == data) {
		cpu->cc = 0x00; /* CC=00 */
	} else {
		cpu->cc = 0x80; /* CC=10 */
	}
}

void
s2650_DAR()
{
	/* * [2650}jA(p.89) DECIMAL ADJUST REGISTER Q]
	 * - DAŘʁACCtOsƂȂ邱ƈȊOPSW͕ω܂B
	 *   CtO̕ωAʃoCgւ̌オ肪܂B
	 *   svcȎdlłAǂӂɗp̂ł傤...?
	 * - dlCCtOsłA݂͕̎ωƂ܂B
	 */
	int hi;
	int lo;
	(*seq++)(); /* Register */
	hi = lo = *reg;
	if(!(cpu->c & 0x100)) {
		hi += 0xa0;
	}
	if(!(cpu->idc & 0x10)) {
		lo += 0x0a;
	}
	*reg = (hi & 0xf0) | (lo & 0x0f);
}

void
s2650_HALT()
{
	cpu->flags |= S2650_HALT;
	cpu->cycle = 0;
}

#endif /*PIECE==============================================================*/

/****************************************************************************
 *	
 ****************************************************************************/
