/*
 *	bmlcnv - BulletML XML to C converter
 *	Copyright (C) 2007 Naoyuki Sawa
 *
 *	* Mon Feb 26 16:34:19 JST 2007 Naoyuki Sawa
 *	- 1st [XB
 */
#include "app.h"

#define VERSION	"20070226"

/*****************************************************************************
 *	
 *****************************************************************************/

const char* get_node_symbol(BulletMLNode* node);
const char* get_value_symbol(BulletMLValue* value);
const char* get_label_symbol(StdString* label);
void reg_node(BulletMLNode* node);
void reg_value(BulletMLValue* value);
void reg_label(StdString* label);
void declare_node(BulletMLNode* node);
void declare_value(BulletMLValue* value);
void declare_label(StdString* label);
void define_node(BulletMLNode* node);
void define_value(BulletMLValue* value);
void define_label(StdString* label);

/*****************************************************************************
 *	
 *****************************************************************************/

/*{{clipfix.c,framfix.c*/
fixed frnd(int* seed) {
	return (fixed)(rnd32(seed) & FRACT_MASK);
}
fixed fmul(fixed a, fixed b) {
	__int64 aa, bb, cc;
	aa = (int)a;
	bb = (int)b;
	cc = (aa * bb) >> FRACT_BITS;
	return (fixed)(int)cc;
}
fixed fdiv(fixed a, fixed b) {
	__int64 aa, bb, cc;
	aa = (int)a;
	bb = (int)b;
	cc = (aa << FRACT_BITS) / bb;
	return (fixed)(int)cc;
}
/*}}clipfix.c,framfix.c*/

/*****************************************************************************
 *	
 *****************************************************************************/

const char* const name_table[BULLETMLNODE_NAME_COUNT] = {
	"BULLETMLNODE_NAME_BULLETML",
	"BULLETMLNODE_NAME_BULLET",
	"BULLETMLNODE_NAME_ACTION",
	"BULLETMLNODE_NAME_FIRE",
	"BULLETMLNODE_NAME_CHANGEDIRECTION",
	"BULLETMLNODE_NAME_CHANGESPEED",
	"BULLETMLNODE_NAME_ACCEL",
	"BULLETMLNODE_NAME_WAIT",
	"BULLETMLNODE_NAME_VANISH",
	"BULLETMLNODE_NAME_REPEAT",
	"BULLETMLNODE_NAME_DIRECTION",
	"BULLETMLNODE_NAME_SPEED",
	"BULLETMLNODE_NAME_HORIZONTAL",
	"BULLETMLNODE_NAME_VERTICAL",
	"BULLETMLNODE_NAME_TERM",
	"BULLETMLNODE_NAME_TIMES",
	"BULLETMLNODE_NAME_BULLETREF",
	"BULLETMLNODE_NAME_ACTIONREF",
	"BULLETMLNODE_NAME_FIREREF",
	"BULLETMLNODE_NAME_PARAM",
};

const char* const type_table[BULLETMLNODE_TYPE_COUNT] = {
	"BULLETMLNODE_TYPE_NONE",
	"BULLETMLNODE_TYPE_VERTICAL",
	"BULLETMLNODE_TYPE_HORIZONTAL",
	"BULLETMLNODE_TYPE_AIM",
	"BULLETMLNODE_TYPE_ABSOLUTE",
	"BULLETMLNODE_TYPE_RELATIVE",
	"BULLETMLNODE_TYPE_SEQUENCE",
};

const char* const id_table[BULLETML_MAX_PARAMS + 2] = {
	"$rank",	/* -2 */
	"$rand",	/* -1 */
	"$1",		/*  0 */
	"$2",		/*  1 */
	"$3",		/*  2 */
	"$4",		/*  3 */
	"$5",		/*  4 */
	"$6",		/*  5 */
	"$7",		/*  6 */
	"$8",		/*  7 */
	"$9",		/*  8 */
};

const char* prefix;
char xml[1024 * 1024];

#define MAX_NODE	10000	/*  */
int node_count;
BulletMLNode* node_array[MAX_NODE];

#define MAX_VALUE	10000	/*  */
int value_count;
BulletMLValue* value_array[MAX_VALUE];

#define MAX_LABEL	10000	/*  */
int label_count;
StdString* label_array[MAX_LABEL];

/*****************************************************************************
 *	
 *****************************************************************************/

const char*
get_node_symbol(BulletMLNode* node)
{
	static char buf[256];
	int i;

	for(i = 0; i < node_count; i++) {
		if(node_array[i] == node) {
			sprintf(buf, "%s_node%d", prefix, i);
			return buf;
		}
	}

	DIE(); /* o^ (L蓾܂BvÕoO?) */
}

const char*
get_value_symbol(BulletMLValue* value)
{
	static char buf[256];
	int i;

	for(i = 0; i < value_count; i++) {
		if(value_array[i] == value) {
			sprintf(buf, "%s_value%d", prefix, i);
			return buf;
		}
	}

	DIE(); /* o^ (L蓾܂BvÕoO?) */
}

const char*
get_label_symbol(StdString* label)
{
	static char buf[256];
	int i;

	for(i = 0; i < label_count; i++) {
		if(!strcmp(StdString_c_str(label_array[i]), StdString_c_str(label))) {
			sprintf(buf, "%s_label%d", prefix, i);
			return buf;
		}
	}

	DIE(); /* o^ (L蓾܂BvÕoO?) */
}

/*****************************************************************************
 *	
 *****************************************************************************/

void
reg_node(BulletMLNode* node)
{
	int i;

	if(node_count >= MAX_NODE) {
		DIE(); /*  (炩P/ECE̋eʃI[o[) */
	}
	for(i = 0; i < node_count; i++) {
		if(node_array[i] == node) {
			DIE(); /* do^ (L蓾܂BvÕoO?) */
		}
	}
	node_array[node_count++] = node;

	if(node->label) {
		reg_label(node->label);
	}
	if(node->value) {
		reg_value(node->value);
	}
	if(node->firstChild) {
		reg_node(node->firstChild);
	}
	if(node->nextSibling) {
		reg_node(node->nextSibling);
	}
}

void
reg_value(BulletMLValue* value)
{
	int i;
	const void* vptr;

	if(value_count >= MAX_VALUE) {
		DIE(); /*  (炩P/ECE̋eʃI[o[) */
	}
	for(i = 0; i < value_count; i++) {
		if(value_array[i] == value) {
			DIE(); /* do^ (L蓾܂BvÕoO?) */
		}
	}
	value_array[value_count++] = value;

	vptr = value->vptr;
	if(vptr == &_BulletMLNumber_vtbl) {
		BulletMLNumber* number = (BulletMLNumber*)value;
	} else if(vptr == &_BulletMLVar_vtbl) {
		BulletMLVar* var = (BulletMLVar*)value;
	} else if(vptr == &_BulletMLUnExpr_vtbl) {
		BulletMLUnExpr* unExpr = (BulletMLUnExpr*)value;
		reg_value(unExpr->value);
	} else if(vptr == &_BulletMLBinExpr_vtbl) {
		BulletMLBinExpr* binExpr = (BulletMLBinExpr*)value;
		reg_value(binExpr->lhs);
		reg_value(binExpr->rhs);
	} else {
		DIE(); /* sBulletMLValuẽTu^Cv (vÕoO?) */
	}
}

void
reg_label(StdString* label)
{
	int i;

	if(label_count >= MAX_LABEL) {
		DIE(); /*  (炩P/ECE̋eʃI[o[) */
	}
	for(i = 0; i < label_count; i++) {
		if(!strcmp(StdString_c_str(label_array[i]), StdString_c_str(label))) {
			return;
		}
	}
	label_array[label_count++] = label;
}

/*****************************************************************************
 *	
 *****************************************************************************/

void
declare_node(BulletMLNode* node)
{
	printf("static BulletMLNode %s;\n", get_node_symbol(node));
}

void
declare_value(BulletMLValue* value)
{
	const void* vptr;

	vptr = value->vptr;
	if(vptr == &_BulletMLNumber_vtbl) {
		printf("static BulletMLNumber %s;\n", get_value_symbol(value));
	} else if(vptr == &_BulletMLVar_vtbl) {
		printf("static BulletMLVar %s;\n", get_value_symbol(value));
	} else if(vptr == &_BulletMLUnExpr_vtbl) {
		printf("static BulletMLUnExpr %s;\n", get_value_symbol(value));
	} else if(vptr == &_BulletMLBinExpr_vtbl) {
		printf("static BulletMLBinExpr %s;\n", get_value_symbol(value));
	} else {
		DIE(); /* sBulletMLValuẽTu^Cv (vÕoO?) */
	}
}

void
declare_label(StdString* label)
{
	printf("static StdString %s;\n", get_label_symbol(label));
}

/*****************************************************************************
 *	
 *****************************************************************************/

void
define_node(BulletMLNode* node)
{
	//printf("static BulletMLNode %s = {\n", get_node_symbol(node));
	//<bulletml>m[h̓O[oɂ܂B
	if(node == node_array[0]) {
		printf("BulletMLNode %s = {\n", prefix);
	} else {
		printf("static BulletMLNode %s = {\n", get_node_symbol(node));
	}
	printf("\t%s,\n", name_table[node->name]);
	printf("\t%s,\n", type_table[node->type]);
	if(node->label) {
		printf("\t&%s,\n", get_label_symbol(node->label));
	} else {
		printf("\tNULL,\n");
	}
	if(node->ref) {
		printf("\t&%s,\n", get_node_symbol(node->ref));
	} else {
		printf("\tNULL,\n");
	}
	if(node->value) {
		printf("\t(BulletMLValue*)&%s,\n", get_value_symbol(node->value));
	} else {
		printf("\tNULL,\n");
	}
	if(node->firstChild) {
		printf("\t&%s,\n", get_node_symbol(node->firstChild));
	} else {
		printf("\tNULL,\n");
	}
	if(node->nextSibling) {
		printf("\t&%s,\n", get_node_symbol(node->nextSibling));
	} else {
		printf("\tNULL,\n");
	}
	printf("};\n");
}

void
define_value(BulletMLValue* value)
{
	const void* vptr;

	vptr = value->vptr;
	if(vptr == &_BulletMLNumber_vtbl) {
		BulletMLNumber* number = (BulletMLNumber*)value;
		printf("static BulletMLNumber %s = {\n", get_value_symbol(value));
		printf("\t&_BulletMLNumber_vtbl,\n");
		printf("\t(fixed)%d, /* %f */\n", number->value, fst(number->value));
		printf("};\n");
	} else if(vptr == &_BulletMLVar_vtbl) {
		BulletMLVar* var = (BulletMLVar*)value;
		printf("static BulletMLVar %s = {\n", get_value_symbol(value));
		printf("\t&_BulletMLVar_vtbl,\n");
		printf("\t%d, /* %s */\n", var->id, id_table[var->id + 2]);
		printf("};\n");
	} else if(vptr == &_BulletMLUnExpr_vtbl) {
		BulletMLUnExpr* unExpr = (BulletMLUnExpr*)value;
		printf("static BulletMLUnExpr %s = {\n", get_value_symbol(value));
		printf("\t&_BulletMLUnExpr_vtbl,\n");
		printf("\t'%c',\n", unExpr->op);
		printf("\t(BulletMLValue*)&%s,\n", get_value_symbol(unExpr->value));
		printf("};\n");
	} else if(vptr == &_BulletMLBinExpr_vtbl) {
		BulletMLBinExpr* binExpr = (BulletMLBinExpr*)value;
		printf("static BulletMLBinExpr %s = {\n", get_value_symbol(value));
		printf("\t&_BulletMLBinExpr_vtbl,\n");
		printf("\t'%c',\n", binExpr->op);
		printf("\t(BulletMLValue*)&%s,\n", get_value_symbol(binExpr->lhs));
		printf("\t(BulletMLValue*)&%s,\n", get_value_symbol(binExpr->rhs));
		printf("};\n");
	} else {
		DIE(); /* sBulletMLValuẽTu^Cv (vÕoO?) */
	}

}

void
define_label(StdString* label)
{
	printf("static StdString %s = {\n", get_label_symbol(label));
	printf("\t\"%s\",\n", StdString_c_str(label));
	printf("};\n");
}

/****************************************************************************
 *	usage
 ****************************************************************************/

void
usage()
{
	fprintf(stderr, "bmlcnv - BulletML XML to C converter (%s)\n", VERSION);
	fprintf(stderr, "Copyright (C) 2007 Naoyuki Sawa\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "USAGE:\n");
	fprintf(stderr, "    bmlcnv prefix < input.xml > output.c\n");
	exit(1);
}

/*****************************************************************************
 *	
 *****************************************************************************/

int
main(int argc, char* argv[])
{
	int i;
	BulletMLNode* bulletml;

	if(argc != 1 + 1) {
		usage();
	}
	prefix = argv[1];

	while(gets(&xml[strlen(xml)])) { /** no job **/ }
	bulletml = BulletML_load(xml);

	reg_node(bulletml);
	//for(i = 0; i < node_count; i++) {
	//<bulletml>m[h̓O[oɂ܂B
	printf("extern BulletMLNode %s;\n", prefix);
	for(i = 1; i < node_count; i++) {
		declare_node(node_array[i]);
	}
	for(i = 0; i < value_count; i++) {
		declare_value(value_array[i]);
	}
	for(i = 0; i < label_count; i++) {
		declare_label(label_array[i]);
	}
	for(i = 0; i < node_count; i++) {
		define_node(node_array[i]);
	}
	for(i = 0; i < value_count; i++) {
		define_value(value_array[i]);
	}
	for(i = 0; i < label_count; i++) {
		define_label(label_array[i]);
	}

	BulletMLNode_delete(bulletml);

	return 0;
}

