/*	
 *	clipw65.c
 *
 *	P/ECE W65C02 Emulator
 *
 *	CLiP - Common Library for P/ECE
 *	Copyright (C) 2001-2005 Naoyuki Sawa
 *
 *	* Sun Apr 10 06:09:00 JST 2005 Naoyuki Sawa
 *	- 쐬JnB
 */
#include "clip.h"

/* W65C02_TRACEV{`ƁAg[Xo͂s܂B */
//#define W65C02_TRACE

/****************************************************************************
 *	O[oϐ
 ****************************************************************************/

#ifndef PIECE /*============================================================*/
W65C02 w65c02; /* P/ECȄꍇ͍RAMɔzu -> framw65a.s */
#endif /*PIECE==============================================================*/

#include "w65c02/seq1.h"
const W65C02SEQ w65c02seq[0x100] = { /* V[PXe[u */
#include "w65c02/seq2.h"
};

/****************************************************************************
 *	[Jϐ
 ****************************************************************************/

#define cpu (&w65c02)
#ifndef PIECE /*============================================================*/
static int addr;
static int data;
static void (* const * pseq)();
#ifdef W65C02_TRACE
static const char* const w65c02mne[0x100] = { /* j[jbNe[u */
#include "w65c02/mne.h"
};
#endif /*W65C02_TRACE*/
#endif /*PIECE==============================================================*/

/****************************************************************************
 *	֐
 ****************************************************************************/

void
w65c02_err()
{
	die("w65c02_error(%04x)", w65c02.pc);
}

void
w65c02_adcd(int n)
{
	int p = w65c02_get_p();
	int a = cpu->a;
	//
	int c;
	int lo;
	int hi;

	c = p & 1;
	lo = (a & 0x0f) + (n & 0x0f) + c;
	hi = (a & 0xf0) + (n & 0xf0);
	if(lo > 0x09) {
		lo += 0x06;
		hi += 0x10;
	}
	p = (p & ~0xc3) |
	    (hi & 0x80) |				/* N */
	    ((~(a ^ n) & (a ^ hi) & 0x80) >> 1);	/* V */
	if(hi > 0x90) {
		hi += 0x60;
	}
	a = (lo & 0x0f) | (hi & 0xf0);
	p |= (!a << 1) |				/* Z */
	     ((hi >> 8) & 1);				/* C */

	cpu->a = a;
	w65c02_set_p(p);
}

void
w65c02_sbcd(int n)
{
	int p = w65c02_get_p();
	int a = cpu->a;
	//
	int c;
	int lo;
	int hi;

	c = ~p & 1;
	lo = (a & 0x0f) - (n & 0x0f) - c;
	hi = (a & 0xf0) - (n & 0xf0);
	if(lo & 0x10) {
		lo -= 0x06;
		hi -= 0x10;
	}
	p = (p & ~0xc3) |
	    (hi & 0x80) |				/* N */
	    (((a ^ n) & (a ^ hi) & 0x80) >> 1);		/* V */
	if(hi & 0x100) {
		hi -= 0x60;
	}
	a = (lo & 0x0f) | (hi & 0xf0);
	p |= (!a << 1) |				/* Z */
	     (~(hi >> 8) & 1);				/* C */

	cpu->a = a;
	w65c02_set_p(p);
}

#ifndef PIECE /*============================================================*/

static void
w65c02_push(int data)
{
	w65c02_write(0x100 | cpu->s, data);
	cpu->s--;
}

static int
w65c02_pull()
{
	cpu->s++;
	return w65c02_read(0x100 | cpu->s);
}

static void
w65c02_interrupt(int brk, int vector)
{
	int p;

	/* BRKȂ΁A1oCg΂܂B(dl) */
	cpu->pc += brk;

	/* PCWX^X^bNɐς݂܂B */
	w65c02_push(HIBYTE(cpu->pc));	/* Push HI(PC) */
	w65c02_push(LOBYTE(cpu->pc));	/* Push LO(PC) */

	/* PWX^l擾܂B */
	p  = w65c02_get_p();	/* NV11DIZC */
	p ^= (brk ^ 1) << 4;	/* NV1BDIZC */

	/* PWX^lX^bNɐς݂܂B */
	w65c02_push(p);		/* Push P */

	/* 荞݃xN^ǂ݁APCݒ肵܂B */
	cpu->pc = w65c02_read(vector + 0) |
		  w65c02_read(vector + 1) << 8;

	/* D=0,I=1 (D=065C02gdl) */
	cpu->di = 0x04;

	/* 荞ݔɂAWAI܂B */
	cpu->wi  &= ~W65C02_WAI;
}

static void
w65c02_check_pending()
{
	if(!(cpu->di & (1<<2)) &&	/* IRQ}XNĂȂāA  */
	    (cpu->wi & W65C02_IRQ)) {	/* IRQvۗĂ... */

		/* IRQvtONA܂B */
		cpu->wi &= ~(W65C02_IRQ);

		/* IRQs܂B */
		w65c02_interrupt(0, 0xfffe);
	}
}

#endif /*PIECE==============================================================*/

/****************************************************************************
 *	AvP[Vp֐
 ****************************************************************************/

int
w65c02_get_p()
{
	int p;

	p  = (cpu->di &   0x0c);		/* ----DI-- */
	p |= (cpu->v  &   0x80) >> 1;		/* -V------ */
	p |= (cpu->c  &   0x01);		/* -------C */
	p |= (cpu->nz &   0xff) ? 0x00 : 0x02;	/* ------Z- */
	p |= (cpu->nz & 0x8080) ? 0x80 : 0x00;	/* N------- */
	p |=              0x30;			/* --11---- */

	return p;
}

void
w65c02_set_p(int p)
{
	cpu->di  = ( p       );			/* ----DI-- */
	cpu->v   = ( p       ) << 1;		/* -V------ */
	cpu->c   = ( p       );			/* -------C */
	cpu->nz  = (~p & 0x02);			/* ------Z- */
	cpu->nz |= ( p       ) << 8;		/* N------- */
}

#ifndef PIECE /*============================================================*/

void
w65c02_reset()
{
	/* ܂NA܂B */
	memset(cpu, 0, sizeof(W65C02));

	/* WX^̏lݒ肵܂B */
	cpu->pc = w65c02_read(0xfffc + 0) |
		  w65c02_read(0xfffc + 1) << 8;
	cpu->s  = 0xff;		/* bBdlł͕s */
	w65c02_set_p(0x04);	/* D=0,I=1 (NVZC=b0Bdlł͕s) */
}

void
w65c02_irq()
{
	/* IRQvtOZbg܂B */
	cpu->wi |= W65C02_IRQ;

	/* IRQs\ׂ܂B */
	w65c02_check_pending();
}

void
w65c02_nmi()
{
	/* NMIs܂B */
	w65c02_interrupt(0, 0xfffa);
}

void
w65c02_run(int cycle)
{
	const W65C02SEQ* seq;
	int opcode;

	/* cTCNi[܂B */
	cpu->cycle = cycle;

	/* WAIȂ΁AcsTCN0ɂāAɃ[v𔲂܂B */
	if(cpu->wi & W65C02_WAI) {
		cpu->cycle = 0; /* (w65c02.cycle=0Ŋ֐𔲂邱ƁBK{!!) */
	}

	/* cTCN0ȉɂȂ܂... */
	while(cpu->cycle > 0) {

		/* OpCodetFb`܂B */
		opcode = w65c02_read(cpu->pc++);
#ifdef W65C02_TRACE
		{
			int p = w65c02_get_p();
			printf("%04x: %-10s A:%02x Y:%02x X:%02x S:%02x P:%c%c%c%c%c%c%c%c (%d)\n",
				cpu->pc - 1, w65c02mne[opcode],
				cpu->a, cpu->y, cpu->x, cpu->s,
				p & (1<<7) ? 'N' : '-',
				p & (1<<6) ? 'V' : '-',
				p & (1<<5) ? '1' : '-',
				p & (1<<4) ? 'B' : '-',
				p & (1<<3) ? 'D' : '-',
				p & (1<<2) ? 'I' : '-',
				p & (1<<1) ? 'Z' : '-',
				p & (1<<0) ? 'C' : '-',
				cpu->cycle);
		}
#endif /*W65C02_TRACE*/

		/* OpCodeɑΉV[PX̐擪AhX擾܂B */
		seq = &w65c02seq[opcode];

		/* csTCN炵܂B(ߎsOɌ炷!!) */
		cpu->cycle -= seq->cycle;

		/* V[PX́Aŏ̊֐Ăт܂B */
		pseq = &seq->fn1;
		(*pseq++)();
	}
}

/****************************************************************************
 *	Action
 ****************************************************************************/

void
w65c02_acc()
{
	data = cpu->a;			/* data = A */
	(*pseq)();			/* data = Operation(data) */
	cpu->a = data;			/* A = data */
}

void
w65c02_imm()
{
	data = w65c02_read(cpu->pc++);	/* data = imm */
	(*pseq)();			/* Operation(data) */
}

void
w65c02_r()
{
	(*pseq++)();			/* addr = AddressMode() */
	data = w65c02_read(addr);	/* data = [addr] */
	(*pseq)();			/* Operation(data) */
}

void
w65c02_w()
{
	(*pseq++)();			/* addr = AddressMode() */
	(*pseq)();			/* data = Operation() */
	w65c02_write(addr, data);	/* [addr] = data */
}

void
w65c02_rw()
{
	(*pseq++)();			/* addr = AddressMode() */
	data = w65c02_read(addr);	/* data = [addr] */
	(*pseq)();			/* data = Operation(data) */
	w65c02_write(addr, data);	/* [addr] = data */
}

static void
w65c02_bcond()
{
	cpu->cycle--;			/* More 1 cycle */
	w65c02_bra();			/* Branch taken */
}

void
w65c02_bra()
{
	data = (char)w65c02_read(cpu->pc++);	/* data = }rel */
	cpu->pc += data;			/* PC = PC}rel */
}

void
w65c02_beq()
{
	(*pseq)();			/* data = Operation() */
	if(data == 0) {
		w65c02_bcond();		/* Branch taken */
	} else {
		cpu->pc++;		/* Skip rel */
	}
}

void
w65c02_bne()
{
	(*pseq)();			/* data = Operation() */
	if(data != 0) {
		w65c02_bcond();		/* Branch taken */
	} else {
		cpu->pc++;		/* Skip rel */
	}
}

void
w65c02_bbr()
{
	w65c02_zp();
	data = w65c02_read(addr);
	w65c02_beq();
}

void
w65c02_bbs()
{
	w65c02_zp();
	data = w65c02_read(addr);
	w65c02_bne();
}

void
w65c02_jmp()
{
	(*pseq)();			/* addr = AddressMode() */
	cpu->pc = addr;			/* PC = addr */
}

void
w65c02_jsr()
{
	(*pseq)();			/* addr = AddressMode() */
	cpu->pc--;			/* (dl) */
	w65c02_push(HIBYTE(cpu->pc));	/* Push HI(PC) */
	w65c02_push(LOBYTE(cpu->pc));	/* Push LO(PC) */
	cpu->pc = addr;			/* PC = addr */
}

/****************************************************************************
 *	AddressMode
 ****************************************************************************/

void
w65c02_zp()
{
	addr = w65c02_read(cpu->pc++);		/* addr = zp */
}

void
w65c02_zp_x()
{
	addr  = w65c02_read(cpu->pc++);		/* addr = zp   */
	addr += cpu->x;				/* addr = zp+X */
	addr  = (unsigned char)addr;		/* ($00..$FF)  */
}

void
w65c02_zp_y()
{
	addr  = w65c02_read(cpu->pc++);		/* addr = zp   */
	addr += cpu->y;				/* addr = zp+Y */
	addr  = (unsigned char)addr;		/* ($00..$FF)  */
}

void
w65c02_a()
{
	addr  = w65c02_read(cpu->pc++);		/* addr = a */
	addr |= w65c02_read(cpu->pc++) << 8;
}

void
w65c02_a_x()
{
	addr  = w65c02_read(cpu->pc++);		/* addr = a       */
	addr |= w65c02_read(cpu->pc++) << 8;
	addr += cpu->x;				/* addr = a+X     */
	addr  = (unsigned short)addr;		/* ($0000..$FFFF) */
}

void
w65c02_a_y()
{
	addr  = w65c02_read(cpu->pc++);		/* addr = a       */
	addr |= w65c02_read(cpu->pc++) << 8;
	addr += cpu->y;				/* addr = a+Y     */
	addr  = (unsigned short)addr;		/* ($0000..$FFFF) */
}

void
w65c02_IzpI()
{
	addr = w65c02_read(cpu->pc++);		/* addr =  zp  */
	addr = w65c02_read(addr + 0) |		/* addr = (zp) */
		w65c02_read(addr + 1) << 8;
}

void
w65c02_Izp_xI()
{
	addr  = w65c02_read(cpu->pc++);		/* addr =  zp    */
	addr += cpu->x;				/* addr =  zp+X  */
	addr  = (unsigned char)addr;		/* ($00..$FF)    */
	addr  = w65c02_read(addr + 0) |		/* addr = (zp+X) */
		w65c02_read(addr + 1) << 8;

}

void
w65c02_IzpI_y()
{
	addr  = w65c02_read(cpu->pc++);		/* addr =  zp     */
	addr  = w65c02_read(addr + 0) |		/* addr = (zp)    */
		w65c02_read(addr + 1) << 8;
	addr += cpu->y;				/* addr = (zp)+Y  */
	addr  = (unsigned short)addr;		/* ($0000..$FFFF) */
}

void
w65c02_IaI()
{
	addr  = w65c02_read(cpu->pc++);		/* addr =  a  */
	addr |= w65c02_read(cpu->pc++) << 8;
	addr  = w65c02_read(addr + 0) |		/* addr = (a) */
		w65c02_read(addr + 1) << 8;
}

void
w65c02_Ia_xI()
{
	addr  = w65c02_read(cpu->pc++);		/* addr =  a      */
	addr |= w65c02_read(cpu->pc++) << 8;
	addr += cpu->x;				/* addr =  a+X    */
	addr  = (unsigned short)addr;		/* ($0000..$FFFF) */
	addr  = w65c02_read(addr + 0) |		/* addr = (a+X)   */
		w65c02_read(addr + 1) << 8;
}

/****************************************************************************
 *	Operation
 ****************************************************************************/

void
w65c02_ERR()
{
	w65c02_err();
}

/*--------------------------------------------------------------------------*/

void
w65c02_LDA()
{
	cpu->a = data;
	cpu->nz = cpu->a;
}

void
w65c02_LDX()
{
	cpu->x = data;
	cpu->nz = cpu->x;
}

void
w65c02_LDY()
{
	cpu->y = data;
	cpu->nz = cpu->y;
}

/*--------------------------------------------------------------------------*/

void
w65c02_STA()
{
	data = cpu->a;
}

void
w65c02_STX()
{
	data = cpu->x;
}

void
w65c02_STY()
{
	data = cpu->y;
}

void
w65c02_STZ()
{
	data = 0;
}

/*--------------------------------------------------------------------------*/

void
w65c02_TXA()
{
	cpu->a = cpu->x;
	cpu->nz = cpu->a;
}

void
w65c02_TAX()
{
	cpu->x = cpu->a;
	cpu->nz = cpu->x;
}

void
w65c02_TYA()
{
	cpu->a = cpu->y;
	cpu->nz = cpu->a;
}

void
w65c02_TAY()
{
	cpu->y = cpu->a;
	cpu->nz = cpu->y;
}

void
w65c02_TSX()
{
	cpu->x = cpu->s;
	cpu->nz = cpu->x;
}

void
w65c02_TXS()
{
	cpu->s = cpu->x;
	/* no P affected */
}

/*--------------------------------------------------------------------------*/

void
w65c02_PHA()
{
	w65c02_push(cpu->a);
}

void
w65c02_PHX()
{
	w65c02_push(cpu->x);
}

void
w65c02_PHY()
{
	w65c02_push(cpu->y);
}

void
w65c02_PHP()
{
	int p = w65c02_get_p();
	w65c02_push(p);
}

void
w65c02_PLA()
{
	cpu->a = w65c02_pull();
	cpu->nz = cpu->a;
}

void
w65c02_PLX()
{
	cpu->x = w65c02_pull();
	cpu->nz = cpu->x;
}

void
w65c02_PLY()
{
	cpu->y = w65c02_pull();
	cpu->nz = cpu->y;
}

void
w65c02_PLP()
{
	int p = w65c02_pull();
	w65c02_set_p(p);
	w65c02_check_pending();
}

/*--------------------------------------------------------------------------*/

static void
w65c02_ADCD()
{
	w65c02_adcd(data);
}

void
w65c02_ADC()
{
	if(cpu->di & (1<<3)) {
		w65c02_ADCD();
	} else {
		int c = cpu->c & 1;
		int t = cpu->a + data + c;
		cpu->c = t >> 8;
		cpu->v = ~(cpu->a ^ data) & (cpu->a ^ t);
		cpu->a = t;
		cpu->nz = cpu->a;
	}
}

static void
w65c02_SBCD()
{
	w65c02_sbcd(data);
}

void
w65c02_SBC()
{
	if(cpu->di & (1<<3)) {
		w65c02_SBCD();
	} else {
		int c = ~cpu->c & 1;
		int t = cpu->a - data - c;
		cpu->c = ~t >> 8;
		cpu->v = (cpu->a ^ data) & (cpu->a ^ t);
		cpu->a = t;
		cpu->nz = cpu->a;
	}
}

void
w65c02_CMP()
{
	int t = cpu->a - data;
	cpu->c = ~t >> 8;
	cpu->nz = (unsigned char)t;
}

void
w65c02_CPX()
{
	int t = cpu->x - data;
	cpu->c = ~t >> 8;
	cpu->nz = (unsigned char)t;
}

void
w65c02_CPY()
{
	int t = cpu->y - data;
	cpu->c = ~t >> 8;
	cpu->nz = (unsigned char)t;
}

/*--------------------------------------------------------------------------*/

void
w65c02_INC()
{
	data++;
	data = (unsigned char)data;
	cpu->nz = data;
}

void
w65c02_INX()
{
	cpu->x++;
	cpu->nz = cpu->x;
}

void
w65c02_INY()
{
	cpu->y++;
	cpu->nz = cpu->y;
}

void
w65c02_DEC()
{
	data--;
	data = (unsigned char)data;
	cpu->nz = data;
}

void
w65c02_DEX()
{
	cpu->x--;
	cpu->nz = cpu->x;
}

void
w65c02_DEY()
{
	cpu->y--;
	cpu->nz = cpu->y;
}

/*--------------------------------------------------------------------------*/

void
w65c02_AND()
{
	cpu->a &= data;
	cpu->nz = cpu->a;
}

void
w65c02_ORA()
{
	cpu->a |= data;
	cpu->nz = cpu->a;
}

void
w65c02_EOR()
{
	cpu->a ^= data;
	cpu->nz = cpu->a;
}

void
w65c02_ASL()
{
	data <<= 1;
	cpu->c = data >> 8;
	data = (unsigned char)data;
	cpu->nz = data;
}

void
w65c02_LSR()
{
	cpu->c = data;
	data >>= 1;
	cpu->nz = data;
}

void
w65c02_ROL()
{
	int c = cpu->c & 1;
	data <<= 1;
	cpu->c = data >> 8;
	data = (unsigned char)data;
	data |= c;
	cpu->nz = data;
}

void
w65c02_ROR()
{
	int c = cpu->c & 1;
	cpu->c = data;
	data >>= 1;
	data |= c << 7;
	cpu->nz = data;
}

void
w65c02_BIT()
{
	cpu->v = data << 1;
	cpu->nz = data << 8;
	data &= cpu->a;
	cpu->nz |= data;
}

/*--------------------------------------------------------------------------*/

void
w65c02_TRB()
{
	if(cpu->nz & 0x8080) {
		cpu->nz = 0x8000;
	} else {
		cpu->nz = 0;
	}
	if(data & cpu->a) {
		cpu->nz |= 1;
	}
	data &= ~cpu->a;
}

void
w65c02_TSB()
{
	if(cpu->nz & 0x8080) {
		cpu->nz = 0x8000;
	} else {
		cpu->nz = 0;
	}
	if(data & cpu->a) {
		cpu->nz |= 1;
	}
	data |= cpu->a;
}

/*--------------------------------------------------------------------------*/

void
w65c02_CLC()
{
	cpu->c &= ~(1<<0);
}

void
w65c02_SEC()
{
	cpu->c |= (1<<0);
}

void
w65c02_CLI()
{
	cpu->di &= ~(1<<2);
	w65c02_check_pending();
}

void
w65c02_SEI()
{
	cpu->di |= (1<<2);
}

void
w65c02_CLD()
{
	cpu->di &= ~(1<<3);
}

void
w65c02_SED()
{
	cpu->di |= (1<<3);
}

void
w65c02_CLV()
{
	cpu->v &= ~(1<<7);
}

/*--------------------------------------------------------------------------*/

void
w65c02_BCC()
{
	data = cpu->c & 1;
}

void
w65c02_BCS()
{
	data = cpu->c & 1;
}

void
w65c02_BEQ()
{
	data = cpu->nz & 0xff;
}

void
w65c02_BNE()
{
	data = cpu->nz & 0xff;
}

void
w65c02_BPL()
{
	data = cpu->nz & 0x8080;
}

void
w65c02_BMI()
{
	data = cpu->nz & 0x8080;
}

void
w65c02_BVC()
{
	data = cpu->v & 0x80;
}

void
w65c02_BVS()
{
	data = cpu->v & 0x80;
}

/*--------------------------------------------------------------------------*/

void
w65c02_NOP()
{
	/** no job **/
}

void
w65c02_WAI()
{
	cpu->wi |= W65C02_WAI;
	cpu->cycle = 0;
}

void
w65c02_STP()
{
	cpu->pc--;
}

void
w65c02_BRK()
{
	w65c02_interrupt(1, 0xfffe);
}

void
w65c02_RTS()
{
	cpu->pc  = w65c02_pull();
	cpu->pc |= w65c02_pull() << 8;
	cpu->pc++; /* (dl) */
}

void
w65c02_RTI()
{
	int p = w65c02_pull();
	w65c02_set_p(p);
	cpu->pc  = w65c02_pull();
	cpu->pc |= w65c02_pull() << 8;
	w65c02_check_pending();
}

/*--------------------------------------------------------------------------*/

void
w65c02_BBR0()
{
	data &= (1<<0);
}

void
w65c02_BBR1()
{
	data &= (1<<1);
}

void
w65c02_BBR2()
{
	data &= (1<<2);
}

void
w65c02_BBR3()
{
	data &= (1<<3);
}

void
w65c02_BBR4()
{
	data &= (1<<4);
}

void
w65c02_BBR5()
{
	data &= (1<<5);
}

void
w65c02_BBR6()
{
	data &= (1<<6);
}

void
w65c02_BBR7()
{
	data &= (1<<7);
}

void
w65c02_BBS0()
{
	data &= (1<<0);
}

void
w65c02_BBS1()
{
	data &= (1<<1);
}

void
w65c02_BBS2()
{
	data &= (1<<2);
}

void
w65c02_BBS3()
{
	data &= (1<<3);
}

void
w65c02_BBS4()
{
	data &= (1<<4);
}

void
w65c02_BBS5()
{
	data &= (1<<5);
}

void
w65c02_BBS6()
{
	data &= (1<<6);
}

void
w65c02_BBS7()
{
	data &= (1<<7);
}

/*--------------------------------------------------------------------------*/

void
w65c02_RMB0()
{
	data &= ~(1<<0);
}

void
w65c02_RMB1()
{
	data &= ~(1<<1);
}

void
w65c02_RMB2()
{
	data &= ~(1<<2);
}

void
w65c02_RMB3()
{
	data &= ~(1<<3);
}

void
w65c02_RMB4()
{
	data &= ~(1<<4);
}

void
w65c02_RMB5()
{
	data &= ~(1<<5);
}

void
w65c02_RMB6()
{
	data &= ~(1<<6);
}

void
w65c02_RMB7()
{
	data &= ~(1<<7);
}

void
w65c02_SMB0()
{
	data |= (1<<0);
}

void
w65c02_SMB1()
{
	data |= (1<<1);
}

void
w65c02_SMB2()
{
	data |= (1<<2);
}

void
w65c02_SMB3()
{
	data |= (1<<3);
}

void
w65c02_SMB4()
{
	data |= (1<<4);
}

void
w65c02_SMB5()
{
	data |= (1<<5);
}

void
w65c02_SMB6()
{
	data |= (1<<6);
}

void
w65c02_SMB7()
{
	data |= (1<<7);
}

#endif /*PIECE==============================================================*/

/****************************************************************************
 *	
 ****************************************************************************/
