/*
 *	gnode.h
 *
 *	N-ary Trees - trees of data with any number of branches
 *
 *	* Sat Feb 06 21:09:21 JST 2016 Naoyuki Sawa
 *	- 1st [XB
 *	- GLibN-ary Trees(http://developer.gnome.org/glib/stable/glib-N-ary-Trees.html)قƂǂ̂܂܎g킹Ē܂B
 *	  ׋̂߂ɁA\[Xǂ݂ȂRgt肵܂A͂قƂǕςĂ܂B
 *	  AsvƎv폜ȗ̂ŁAsȓ삪oꍇ͂̕ύX_^ĉB
 */
#ifndef __GNODE_H__
#define __GNODE_H__
#ifdef  __cplusplus
extern "C" {
#endif//__cplusplus
/****************************************************************************
 *	
 ****************************************************************************/
//The GNode struct and its associated functions provide a N-ary tree data structure, where nodes in the tree can contain arbitrary data.
//To create a new tree use g_node_new().
//To insert a node into a tree use g_node_insert(), g_node_insert_before(), g_node_append() and g_node_prepend().
//To create a new node and insert it into a tree use g_node_insert_data(), g_node_insert_data_after(), g_node_insert_data_before(), g_node_append_data() and g_node_prepend_data().
//To reverse the children of a node use g_node_reverse_children().
//To find a node use g_node_get_root(), g_node_find(), g_node_find_child(), g_node_child_index(), g_node_child_position(), g_node_first_child(), g_node_last_child(), g_node_nth_child(), g_node_first_sibling(), g_node_prev_sibling(), g_node_next_sibling() or g_node_last_sibling().
//To get information about a node or tree use G_NODE_IS_LEAF(), G_NODE_IS_ROOT(), g_node_depth(), g_node_n_nodes(), g_node_n_children(), g_node_is_ancestor() or g_node_max_height().
//To traverse a tree, calling a function for each node visited in the traversal, use g_node_traverse() or g_node_children_foreach().
//To remove a node or subtree from a tree use g_node_unlink() or g_node_destroy().
/*--------------------------------------------------------------------------*/
//The GNode struct represents one node in a n-ary tree.
typedef struct _GNode {
	void*			data;		//Contains the actual data of the node.
	struct _GNode*		next;		//Points to the node's next sibling (a sibling is another GNode with the same parent).
	struct _GNode*		prev;		//Points to the node's previous sibling.
	struct _GNode*		parent;		//Points to the parent of the GNode, or is NULL if the GNode is the root of the tree.
	struct _GNode*		children;	//Points to the first child of the GNode. The other children are accessed by using the next pointer of each child.
} GNode;
/*--------------------------------------------------------------------------*/
//Specifies the type of traveral performed by g_tree_traverse(), g_node_traverse() and g_node_find().
//The different orders are illustrated here:
//
//@@@@@e@@@@@
//@@@@^@_@@@@
//@@@^@@@_@@@
//@@a@@@@@f@@
//@^@_@@@@@_@
//`@@@c@@@@@h
//@@@^@_@@@^@
//@@b@@@d@g@@
//
//   In order: A, B, C, D, E, F, G, H, I	ˏȗ܂B񕪖؂ł͂Ȃ̂ł̏ɂ͂܂ӖƎvłBĂcɂȂ̂ŏȗėǂƎv܂B
//  Pre order: F, B, A, D, C, E, G, I, H
// Post order: A, C, E, D, B, H, I, G, F
//Level order: F, B, G, A, D, I, C, E, H
//ȗ	#define G_IN_ORDER	0	//Vists a node's left child first, then the node itself, then its right child. This is the one to use if you want the output sorted according to the compare function.
#define G_PRE_ORDER		1	//Visits a node, then its children.
#define G_POST_ORDER		2	//Visits the node's children, then the node itself.
#define G_LEVEL_ORDER		3	//Is not implemented for balanced binary trees. For n-ary trees, it vists the root node first, then its children, then its grandchildren, and so on. Note that this is less efficient than the other orders.
/*--------------------------------------------------------------------------*/
//Specifies which nodes are visited during several of the tree functions, including g_node_traverse() and g_node_find().
#define G_TRAVERSE_LEAVES	(1<<0)						//Only leaf nodes should be visited. This name has been introduced in 2.6, for older version use G_TRAVERSE_LEAFS.
#define G_TRAVERSE_NON_LEAVES	(1<<1)						//Only non-leaf nodes should be visited. This name has been introduced in 2.6, for older version use G_TRAVERSE_NON_LEAFS.
#define G_TRAVERSE_ALL		(G_TRAVERSE_LEAVES|G_TRAVERSE_NON_LEAVES)	//All nodes should be visited.
#define G_TRAVERSE_MASK		G_TRAVERSE_ALL					//A mask of all traverse flags.
#define G_TRAVERSE_LEAFS	G_TRAVERSE_LEAVES				//Identical to G_TRAVERSE_LEAVES.
#define G_TRAVERSE_NON_LEAFS	G_TRAVERSE_NON_LEAVES				//Identical to G_TRAVERSE_NON_LEAVES.
/*--------------------------------------------------------------------------*/
//Returns TRUE if a GNode is the root of a tree.
//Parameters
//		node		A GNode.
//Returns
//		TRUE if the GNode is the root of a tree (i.e. it has no parent or siblings).
#define G_NODE_IS_ROOT(node)	((!((GNode*)(node))->parent)&& \
				 (!((GNode*)(node))->prev  )&& \
				 (!((GNode*)(node))->next  ))
/*--------------------------------------------------------------------------*/
//Returns TRUE if a GNode is a leaf node.
//Parameters
//		node		A GNode.
//Returns
//		TRUE if the GNode is a leaf node (i.e. it has no children).
#define G_NODE_IS_LEAF(node)	(!((GNode*)(node))->children)
/****************************************************************************
 *	
 ****************************************************************************/
GNode* g_node_new(void* data);
GNode* g_node_copy(GNode* node);
GNode* g_node_copy_deep(GNode* node, void* (*copy_func)(const void* src, void* data), void* data);
GNode* g_node_insert(GNode* parent, int position, GNode* node);
GNode* g_node_insert_before(GNode* parent, GNode* sibling, GNode* node);
GNode* g_node_insert_after(GNode* parent, GNode* sibling, GNode* node);
GNode* g_node_append(GNode* parent, GNode* node);
GNode* g_node_prepend(GNode* parent, GNode* node);
GNode* g_node_insert_data(GNode* parent, int position, void* data);
GNode* g_node_insert_data_after(GNode* parent, GNode* sibling, void* data);
GNode* g_node_insert_data_before(GNode* parent, GNode* sibling, void* data);
GNode* g_node_append_data(GNode* parent, void* data);
GNode* g_node_prepend_data(GNode* parent, void* data);
void g_node_reverse_children(GNode* node);
void g_node_traverse(GNode* root, int order, int flags, int depth, int (*func)(GNode* node, void* data), void* data);
void g_node_children_foreach(GNode* node, int flags, void (*func)(GNode* node, void* data), void* data);
GNode* g_node_get_root(GNode* node);
GNode* g_node_find(GNode* root, int order, int flags, void* data);
GNode* g_node_find_child(GNode* node, int flags, void* data);
int g_node_child_index(GNode* node, void* data);
int g_node_child_position(GNode* node, GNode* child);
GNode* g_node_first_child(GNode* node);
GNode* g_node_last_child(GNode* node);
GNode* g_node_nth_child(GNode* node, int n);
GNode* g_node_first_sibling(GNode* node);
GNode* g_node_next_sibling(GNode* node);
GNode* g_node_prev_sibling(GNode* node);
GNode* g_node_last_sibling(GNode* node);
int g_node_depth(GNode* node);
int g_node_n_nodes(GNode* root, int flags);
int g_node_n_children(GNode* node);
int g_node_is_ancestor(GNode* node, GNode* descendant);
int g_node_max_height(GNode* root);
void g_node_unlink(GNode* node);
void g_node_destroy(GNode* root);
#ifdef  __cplusplus
}//extern "C"
#endif//__cplusplus
#endif//__GNODE_H__
