//
//	clipgrp_uni.cs
//
//	アプリケーションのグラフィック関数
//
//	CLiP - Common Library for P/ECE
//	Copyright (C) 2017 Naoyuki Sawa
//
//	* Mon Mar 27 22:39:32 JST 2017 Naoyuki Sawa
//	- 1st リリース。
//	  まだアプリケーションとライブラリの分離が不完全で、当モジュール内にアプリケーション依存の定数やグローバル変数への参照が残っています。
//	  そのため、現在はまだ、当モジュールを直接アプリケーションに含めて、ビルドしています。
//	  引き続き、アプリケーションとライブラリの分離を行って行きます。(※TODO:)
//
#if     UNITY_5_3_OR_NEWER
using System;
using System.IO;
using System.Collections.Generic;
using System.Reflection;
using UnityEngine;
using UnityEngine.SceneManagement;
using static org.piece_me.libclip;
using static Const;
public static partial class Program {
	//*****************************************************************************
	//	ローカル変数
	//*****************************************************************************
	private static class Graph {
		public static GameObject			goRoot;			//表示オブジェクトのルートGameObject
		public static int				layer;			//オブジェクトプールから次に取得するオブジェクトに設定するレイヤー	Reset()で戻す。CbkQue_SetLayer()で設定する。
		public static int				sortingOrder;		//オブジェクトプールから次に取得するオブジェクトに設定するオーダー順	Reset()で戻す。CbkQue_SetSortingOrder()で明示的に設定する。ObjectPool<T>.GetObject()で自動的に進む。
		//-----------------------------------------------------------------------------
		public static ObjectPool<MovieObject>		movieObjectPool		= new ObjectPool<MovieObject>();	//Movieオブジェクトプール
		public static ObjectPool<SpriteObject>		spriteObjectPool	= new ObjectPool<SpriteObject>();	//Spriteオブジェクトプール
		public static ObjectPool<LineObject>		lineObjectPool		= new ObjectPool<LineObject>();		//Lineオブジェクトプール
		//-----------------------------------------------------------------------------
		public static Dictionary<int,ST_TexDef>		texDefCache		= new Dictionary<int,ST_TexDef>();	//取得済みのテクスチャ定義キャッシュ
		public static Dictionary<int,ST_SprDef>		sprDefCache		= new Dictionary<int,ST_SprDef>();	//取得済みのスプライト定義キャッシュ
		public static Dictionary<int,Sprite>		spriteCache		= new Dictionary<int,Sprite>();		//ロード済みのスプライトキャッシュ
		//-----------------------------------------------------------------------------
		public static ST_MovieHelper			pMovieHelper;
	}
	//*****************************************************************************
	//	オブジェクトプール
	//*****************************************************************************
	private class ObjectPool<T> where T : PoolObject, new() {
		public List<T>		TBL_PoolObject	= new List<T>();	//作成した全てのオブジェクト
		public int		Count;					//オブジェクトを取り出した数
		//-----------------------------------------------------------------------------
		//オブジェクトを取得する。
		public T GetObject() {
			//プールが空ならば…
			if(Count == TBL_PoolObject.Count) {
				//オブジェクトを作成して、プールに追加する。
				TBL_PoolObject.Add(new T());
			}
			//プールから、オブジェクトを取得する。
			T po = TBL_PoolObject[Count++];
			//オブジェクトを有効にする。
			po.SetActive(true);
			//オブジェクトのレイヤを設定する。
			po.SetLayer(Graph.layer);
			//レンダラーのオーダー順を設定する。
			po.SetSortingOrder(Graph.sortingOrder++);
			//オブジェクトを返す。
			return po;
		}
		//-----------------------------------------------------------------------------
		//プールをリセットする。
		public void Reset() {
			//取り出された各オブジェクトについて…
			while(Count != 0) {
				//オブジェクトを無効に戻す。
				T po = TBL_PoolObject[--Count];
				po.SetActive(false);
			}
		}
	}
	//*****************************************************************************
	//	プールオブジェクトの基底クラス
	//*****************************************************************************
	private abstract class PoolObject {
		public GameObject		goOuter;		//外側のオブジェクト
		public GameObject		goInner;		//内側のオブジェクト
		public abstract Renderer	renderer { get; }	//内側のオブジェクトのレンダラー
		//-----------------------------------------------------------------------------
		protected PoolObject() {
			//外側のGameObjectを作成する。
			goOuter = new GameObject();
			goOuter.transform.SetParent(Graph.goRoot.transform, false);
		}
		//-----------------------------------------------------------------------------
		public void SetActive(bool value) {
			//外側のオブジェクトを有効,又は,無効にする。
			goOuter.SetActive(value);
		}
		//-----------------------------------------------------------------------------
		public void SetLayer(int layer) {
			//内側のオブジェクトのレイヤを設定する。
			goInner.layer = layer;
		}
		//-----------------------------------------------------------------------------
		public void SetSortingOrder(int sortingOrder) {
			//レンダラーのオーダー順を設定する。
			renderer.sortingOrder = sortingOrder;
		}
	}
	//*****************************************************************************
	//	Movieオブジェクト
	//*****************************************************************************
	private class MovieObject : PoolObject {
		public MeshRenderer	meshRenderer;		//内側のオブジェクトのMeshRenderer
		//-----------------------------------------------------------------------------
		public MovieObject() {
			//内側のGameObjectを作成する。
			goInner = GameObject.CreatePrimitive(PrimitiveType.Quad);
			goInner.transform.SetParent(goOuter.transform, false);
			//MeshRendererを取得しておく。
			meshRenderer = goInner.GetComponent<MeshRenderer>();
			if(meshRenderer == null) { throw new ApplicationException(); }
		}
		//-----------------------------------------------------------------------------
		//PoolObject
		public override Renderer renderer {
			get {
				//内側のオブジェクトのレンダラーを返す。
				return meshRenderer;
			}
		}
	}
	//*****************************************************************************
	//	Spriteオブジェクト
	//*****************************************************************************
	private class SpriteObject : PoolObject {
		public SpriteRenderer	spriteRenderer;		//内側のオブジェクトのSpriteRenderer
		//-----------------------------------------------------------------------------
		public SpriteObject() {
			//内側のGameObjectを作成する。
			goInner = new GameObject(null, typeof(SpriteRenderer));
			goInner.transform.SetParent(goOuter.transform, false);
			//SpriteRendererを取得しておく。
			spriteRenderer = goInner.GetComponent<SpriteRenderer>();
			if(spriteRenderer == null) { throw new ApplicationException(); }
		}
		//-----------------------------------------------------------------------------
		//PoolObject
		public override Renderer renderer {
			get {
				//内側のオブジェクトのレンダラーを返す。
				return spriteRenderer;
			}
		}
	}
	//*****************************************************************************
	//	Lineオブジェクト
	//*****************************************************************************
	private class LineObject : PoolObject {
		public LineRenderer	lineRenderer;		//内側のオブジェクトのLineRenderer
		public Material		material;		//内側のオブジェクトのLineRendererのMaterial
		//-----------------------------------------------------------------------------
		public LineObject() {
			//内側のGameObjectを作成する。
			goInner = new GameObject();
			goInner.transform.SetParent(goOuter.transform, false);
			goInner.transform.localPosition = new Vector2(	//位置は固定。
				(float)(-(DISP_X / 2) / PPU),
				(float)( (DISP_Y / 2) / PPU));
			goInner.transform.localScale = new Vector2(	//スケールも固定。ただし、goInner.transform.localScaleはlineRenderer.startWidth,endWidthには影響しないようなので、lineRenderer.startWidth,endWidthにもスケールを掛ける必要が有ります。
				(float)( 1.0 / PPU),
				(float)(-1.0 / PPU));
			//LineRendererを作成する。
			lineRenderer = goInner.AddComponent<LineRenderer>();
			lineRenderer.useWorldSpace = false;	//必要です!!
			//Materialを作成する。
			material = new Material(Shader.Find("Sprites/Default"));	//※要検討。深度バッファを無視して背面消去もしないシェーダなら何でも良い。	【注意】シーンでこのシェーダが参照されない場合は、ProjectSettings/Graphicsで"Always Included Shaders"リストに含めておく必要が有ります。現在の所、シーンでこのシェーダが参照しているので、左記の設定は必要有りません。
			if(material == null) { throw new ApplicationException(); }
			lineRenderer.material = material;
		}
		//-----------------------------------------------------------------------------
		//PoolObject
		public override Renderer renderer {
			get {
				//内側のオブジェクトのレンダラーを返す。
				return lineRenderer;
			}
		}
	}
	//*****************************************************************************
	//	テクスチャ定義
	//*****************************************************************************
	public class ST_TexDef {
		public string		sPath;			//パス			スプライトの場合はリソースパス、ムービーの場合はファイルパス
		public bool		bLoop;			//ループフラグ		ムービーの場合のみ
		public bool		bAdditiveMode;		//加算モード
		//-----------------------------------------------------------------------------
		public bool	isSprite{
			get {
				return string.IsNullOrEmpty(Path.GetExtension(sPath));	//※仮:パスに拡張子が無ければ、リソースパスであると判断して、スプライトと見なす事にした。
			}
		}
		//-----------------------------------------------------------------------------
		public bool	isMovie {
			get {
				return !isSprite;					//※仮:スプライトでなければ、ムービーと見なす事にした。
			}
		}
	}
	//*****************************************************************************
	//	スプライト定義
	//*****************************************************************************
	public class ST_SprDef {
		public int		iTexNo;			//テクスチャ番号
		public int		x, y;			//左上座標
		public int		w, h;			//幅、高さ
		public int		cx, cy;			//原点オフセット
		//-----------------------------------------------------------------------------
		public bool	isSprite{
			get {
				return GetTexDef(iTexNo).isSprite;
			}
		}
		//-----------------------------------------------------------------------------
		public bool	isMovie {
			get {
				return GetTexDef(iTexNo).isMovie;
			}
		}
	}
	//*****************************************************************************
	//	Movieプレイヤーの実装
	//*****************************************************************************
	public class MoviePlayer : IMoviePlayer {
		public CriManaMovieMaterial	criManaMovieMaterial;	//ムービーマテリアル
		//-----------------------------------------------------------------------------
		public MoviePlayer() {
			//ムービーマテリアルを作成する。
			criManaMovieMaterial = Graph.goRoot.AddComponent<CriManaMovieMaterial>();
		}
		//-----------------------------------------------------------------------------
		void IDisposable.Dispose() {
			//ムービーマテリアルを削除する。
			UnityEngine.Object.Destroy(criManaMovieMaterial);
		}
		//-----------------------------------------------------------------------------
		MoviePlayerStatus IMoviePlayer.GetStatus() {
			//プレイヤーの状態を取得する。
			return (MoviePlayerStatus)criManaMovieMaterial.player.status;
		}
		//-----------------------------------------------------------------------------
		void IMoviePlayer.Prepare(int iTexNo) {
			//テクスチャ番号を検査する。
			if((iTexNo < 1) || (iTexNo > TexNo_Max)) { throw new ApplicationException(); }
			//テクスチャ情報を取得する。
			ST_TexDef stTexDef = GetTexDef(iTexNo);
			//テクスチャのファイルパス,ループ設定,加算合成モード設定を設定する。
			if(!criManaMovieMaterial.player.SetFile(binder, stTexDef.sPath)) { throw new ApplicationException(); }
			criManaMovieMaterial.player.Loop(stTexDef.bLoop);
			criManaMovieMaterial.player.additiveMode = stTexDef.bAdditiveMode;
			//再生準備を開始する。
			criManaMovieMaterial.player.Prepare();
		}
		//-----------------------------------------------------------------------------
		void IMoviePlayer.Start() {
			//再生を開始する。
			criManaMovieMaterial.player.Start();
			//マテリアルが有効になるまで待つ。
			for(;;) {
				if(criManaMovieMaterial.player.status == CriMana.Player.Status.Error) { throw new ApplicationException(); }
				if(criManaMovieMaterial.isMaterialAvailable) { break; }
				typeof(CriManaMovieMaterial).InvokeMember("Update", BindingFlags.InvokeMethod|BindingFlags.NonPublic, null, criManaMovieMaterial, null);	//CriManaMovieMaterial.Update()を呼び出すために左記の処理が必要です。criManaMovieMaterial.player.Update()ではマテリアルが更新されないのでダメです(Unityが落ちたりします)。
			}
		}
		//-----------------------------------------------------------------------------
		void IMoviePlayer.Stop() {
			//再生,又は,再生準備を停止する。
			criManaMovieMaterial.Stop();
			//プレイヤーの状態が停止中になるまで待つ。
			for(;;) {
				CriMana.Player.Status status = criManaMovieMaterial.player.status;
				if(status == CriMana.Player.Status.Error) { throw new ApplicationException(); }
				if(status == CriMana.Player.Status.Stop) { break; }
				typeof(CriManaMovieMaterial).InvokeMember("Update", BindingFlags.InvokeMethod|BindingFlags.NonPublic, null, criManaMovieMaterial, null);	//CriManaMovieMaterial.Update()を呼び出すために左記の処理が必要です。criManaMovieMaterial.player.Update()ではマテリアルが更新されないのでダメです(Unityが落ちたりします)。
			}
		}
	}
	//*****************************************************************************
	//	グローバル関数
	//*****************************************************************************
	//グラフィックの初期化処理
	public static void Graph_Init() {
		//ルートGameObjectを作成する。
		Graph.goRoot = new GameObject();
	//{{任意:普段はUnityエディタ上で確認する必要が無いと思うので非表示にしておく事にした。Unityエディタ上で確認する場合はこの範囲をコメントアウトして下さい。
		Graph.goRoot.hideFlags = HideFlags.HideInHierarchy;
	//}}任意:普段はUnityエディタ上で確認する必要が無いと思うので非表示にしておく事にした。Unityエディタ上で確認する場合はこの範囲をコメントアウトして下さい。
		//新しいシーンを読み込んでもこのオブジェクトが自動で破壊されないように設定する。
		// - LoadScene()の後にそのフレーム内でGameObjectを作成した場合、DontDestroyOnLoad()を呼んでいなければそのGameObjectも削除されるようだ。
		//   LoadScene()関数の中でシーンが移行するのでなく、そのフレームの最後にシーンが移行しているのかも知れない。(※未検証)
		//   安全のためには、LoadScene()を行ったフレームでは、他の処理は何も行わない方が良さそうだ。
		//   ここでDontDestroyOnLoad()を呼び出す目的は、上記の問題を回避するためではないのだが、結果的に問題を回避出来ている。
		// - ルートGameObjectに対してDontDestroyOnLoad()を呼び出しておけば、その全トランスフォーム階層も破壊されない。
		//   従って、この後で作成してルートGameObjectの子にする個々のGameObjectに対しては、DontDestroyOnLoad()を呼び出す必要は無い。
		UnityEngine.Object.DontDestroyOnLoad(Graph.goRoot);
		//ムービー管理の初期化を実行する。
		int nCh = criWareInitializer.manaConfig.numberOfDecoders / 2;	//全てのムービーマテリアルでアルファムービーを再生してもデコーダが足りるように、デコーダの半分の数とする事にした。
		Graph.pMovieHelper = MovieHelper_New(nCh, delegate(object arg) { return new MoviePlayer(); }, null);
	}
	//-----------------------------------------------------------------------------
	//グラフィックの周期処理
	public static void Graph_Exec() {
		//オブジェクトプールをリセットする。
		Graph.layer = 0/*Default*/;		//オブジェクトプールから次に取得するオブジェクトに設定するレイヤーをDefaultに戻す。
		Graph.sortingOrder = short.MinValue;	//オブジェクトプールから次に取得するオブジェクトに設定するオーダー順を最小値に戻す。
		Graph.movieObjectPool.Reset();		//Movieオブジェクトプールから取り出したオブジェクトを無効にしてプールに返す。
		Graph.spriteObjectPool.Reset();		//Spriteオブジェクトプールから取り出したオブジェクトを無効にしてプールに返す。
		Graph.lineObjectPool.Reset();		//Lineオブジェクトプールから取り出したオブジェクトを無効にしてプールに返す。
		//ムービー管理の周期処理を実行する。
		MovieHelper_Exec(Graph.pMovieHelper);
	}
	//-----------------------------------------------------------------------------
	//グラフィックの終了処理
	public static void Graph_Exit() {
		//ムービー管理の終了処理を実行する。
		MovieHelper_Exit(Graph.pMovieHelper);
		//ルートGameObjectを削除する。
		UnityEngine.Object.Destroy(Graph.goRoot);
	}
	//*****************************************************************************
	//	ローカル関数
	//*****************************************************************************
	//ムービーマテリアルを取得する。
	private static CriManaMovieMaterial GetMovieMaterial(int iTexNo) {
		MoviePlayer moviePlayer = (MoviePlayer)MovieHelper_Draw(Graph.pMovieHelper, iTexNo);
		return moviePlayer.criManaMovieMaterial;
	}
	//-----------------------------------------------------------------------------
	//スプライトを取得する。
	private static Sprite GetSprite(int iSprNo) {
		Sprite sprite;
		//スプライト番号を検査する。
		if((iSprNo < 1) || (iSprNo > SprNo_Max)) { throw new ApplicationException(); }
		//指定された番号のスプライトが、まだロードされていなければ…
		if(!Graph.spriteCache.TryGetValue(iSprNo, out sprite)) {
			//スプライト情報を取得する。
			ST_SprDef stSprDef = GetSprDef(iSprNo);
			//テクスチャ情報を取得する。
			ST_TexDef stTexDef = GetTexDef(stSprDef.iTexNo);
			//スプライトをロードする。
			sprite = Resources.Load<Sprite>(stTexDef.sPath);
			if(sprite == null) { throw new ApplicationException(); }
			//スプライトをキャッシュに登録する。
			Graph.spriteCache.Add(iSprNo, sprite);
		}
		//スプライトを返す。
		return sprite;
	}
	//-----------------------------------------------------------------------------
	//スプライト定義を取得する。
	public static ST_SprDef GetSprDef(int iSprNo) {
		ST_SprDef stSprDef;
		//スプライト番号を検査する。
		if((iSprNo < 1) || (iSprNo > SprNo_Max)) { throw new ApplicationException(); }
		//指定された番号のスプライト定義が、まだ取得されていなければ…
		if(!Graph.sprDefCache.TryGetValue(iSprNo, out stSprDef)) {
			//スプライト定義を取得する。
			VoidPtr pSprDef = REG_open_key_l(TBL_RegTbl, RegKey_SprDef, iSprNo);
			if(!pSprDef) { throw new ApplicationException(); }
			stSprDef = new ST_SprDef() {
				iTexNo = REG_get_value(pSprDef, RegKey_TexNo),
				x      = REG_get_value(pSprDef, RegKey_x),
				y      = REG_get_value(pSprDef, RegKey_y),
				w      = REG_get_value(pSprDef, RegKey_w),
				h      = REG_get_value(pSprDef, RegKey_h),
				cx     = REG_get_value(pSprDef, RegKey_cx),
				cy     = REG_get_value(pSprDef, RegKey_cy),
			};
			if((stSprDef.iTexNo == -1) ||
			   (stSprDef.x      == -1) ||
			   (stSprDef.y      == -1) ||
			   (stSprDef.w      == -1) ||
			   (stSprDef.h      == -1) ||
			   (stSprDef.cx     == -1) ||
			   (stSprDef.cy     == -1)) { throw new ApplicationException(); }
			//スプライト定義をキャッシュに登録する。
			Graph.sprDefCache.Add(iSprNo, stSprDef);
		}
		//スプライト定義を返す。
		return stSprDef;
	}
	//-----------------------------------------------------------------------------
	//テクスチャ定義を取得する。
	private static ST_TexDef GetTexDef(int iTexNo) {
		ST_TexDef stTexDef;
		//テクスチャ番号を検査する。
		if((iTexNo < 1) || (iTexNo > TexNo_Max)) { throw new ApplicationException(); }
		//指定された番号のテクスチャ定義が、まだ取得されていなければ…
		if(!Graph.texDefCache.TryGetValue(iTexNo, out stTexDef)) {
			//テクスチャ定義を取得する。
			VoidPtr pTexDef = REG_open_key_l(TBL_RegTbl, RegKey_TexDef, iTexNo);
			if(!pTexDef) { throw new ApplicationException(); }
			string sPath = REG_get_string(pTexDef, RegKey_Path);
			if(sPath == null) { throw new ApplicationException(); }
			bool bLoop = (REG_get_value(pTexDef, RegKey_loop) == 1);		//ループ無しならばレジストリにループフラグは格納されておらず、REG_get_value_l()の戻り値は(-1)である。ループ有りならばレジストリに'Loop=1'が格納されている。従って、REG_get_value_l()の戻り値が1である場合のみbLoopを1とすれば良い。
			bool bAdditiveMode = (REG_get_value(pTexDef, RegKey_additive) == 1);	//通常モードならばレジストリにモードフラグは格納されておらず、REG_get_value_l()の戻り値は(-1)である。加算モードならばレジストリに'AdditiveMode=1'が格納されている。従って、REG_get_value_l()の戻り値が1である場合のみbAdditiveModeを1とすれば良い。
			stTexDef = new ST_TexDef() {
				sPath = sPath,
				bLoop = bLoop,
				bAdditiveMode = bAdditiveMode,
			};
			//テクスチャ定義をキャッシュに登録する。
			Graph.texDefCache.Add(iTexNo, stTexDef);
		}
		//テクスチャ定義を返す。
		return stTexDef;
	}
	//*****************************************************************************
	//	描画関数
	//*****************************************************************************
	//シーンを切り替える。
	private static void LoadScene(string sceneName) {
		SceneManager.LoadScene(sceneName);
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_LoadScene(ST_CbkQue pCbkQue, int pri, string sceneName) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				LoadScene(sceneName);
			}, null);
	}
	//-----------------------------------------------------------------------------
	//オブジェクトプールから次に取得するオブジェクトに設定するレイヤーを設定する。
	private static void SetLayer(int layer) {
		Graph.layer = layer;
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_SetLayer(ST_CbkQue pCbkQue, int pri, int layer) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				SetLayer(layer);
			}, null);
	}
	//-----------------------------------------------------------------------------
	//オブジェクトプールから次に取得するオブジェクトに設定するオーダー順を設定する。
	private static void SetSortingOrder(int sortingOrder) {
		Graph.sortingOrder = sortingOrder;
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_SetSortingOrder(ST_CbkQue pCbkQue, int pri, int sortingOrder) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				SetSortingOrder(sortingOrder);
			}, null);
	}
	//-----------------------------------------------------------------------------
	//スプライト準備
	private static void prep_spr(int iSprNo) {
		//スプライト情報を取得する。
		ST_SprDef stSprDef = GetSprDef(iSprNo);
		//ムービーならば…
		if(stSprDef.isMovie) {
			//ムービーマテリアルを準備する。
			MovieHelper_Prepare(Graph.pMovieHelper, stSprDef.iTexNo);
		//スプライトならば…
		} else {
			//スプライトは裏読み未対応です。
			/** no job **/
		}
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_prep_spr(ST_CbkQue pCbkQue, int pri, int iSprNo) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				prep_spr(iSprNo);
			}, null);
	}
	//-----------------------------------------------------------------------------
	//スプライト描画(フォールバック有り)
	private static void draw_spr(double x, double y, int[] aSprNo, double scale_x, double scale_y, double rot, double alpha) {
		//各候補について…
		for(int i = 0; i < aSprNo.Length; i++) {	//┬この下の処理で、最終候補かを判定する必要が有るので、foreachではなくforを使ってループする。
			int iSprNo = aSprNo[i];			//┘
			//スプライト情報を取得する。
			ST_SprDef stSprDef = GetSprDef(iSprNo);
			PoolObject po;
			//ムービーならば…
			if(stSprDef.isMovie) {
				//最終候補でなければ…
				if(i < (aSprNo.Length - 1)) {
					//ムービーマテリアルを準備する。
					if(!MovieHelper_Prepare(Graph.pMovieHelper, stSprDef.iTexNo)) { continue; }	//まだ準備が完了していなければ、次の候補へ進む。
				}
				//ムービーマテリアルを取得する。
				CriManaMovieMaterial movieMaterial = ((MoviePlayer)MovieHelper_Draw(Graph.pMovieHelper, stSprDef.iTexNo)).criManaMovieMaterial;
				if(movieMaterial == null) { throw new ApplicationException(); }	//バグ
				//Movieオブジェクトを取得する。
				MovieObject mo = Graph.movieObjectPool.GetObject();
				//Movieオブジェクトを設定する。
				mo.meshRenderer.material = movieMaterial.material;
				//※TODO:alpha未対応
				mo.goInner.transform.localPosition = new Vector2(
					(float)(((stSprDef.w / 2) - (stSprDef.cx)) / PPU),
					(float)(((stSprDef.cy) - (stSprDef.h / 2)) / PPU));
				mo.goInner.transform.localScale = new Vector2(
					(float)(stSprDef.w / PPU),
					(float)(stSprDef.h / PPU));
				//Poolオブジェクトの共通処理へ進む。
				po = mo;
			//スプライトならば…
			} else {
				//スプライトを取得する。
				Sprite sprite = GetSprite(iSprNo);
				//Spriteオブジェクトを取得する。
				SpriteObject so = Graph.spriteObjectPool.GetObject();
				//Spriteオブジェクトを設定する。
				so.spriteRenderer.sprite = sprite;
				so.spriteRenderer.color = new Color(1, 1, 1, (float)alpha);
				so.goInner.transform.localPosition = new Vector2(
					(sprite.pivot.x - stSprDef.cx) / sprite.pixelsPerUnit,
					(stSprDef.cy - sprite.pivot.y) / sprite.pixelsPerUnit);
				//Poolオブジェクトの共通処理へ進む。
				po = so;
			}
			//Poolオブジェクトを設定する。
			po.goOuter.transform.localPosition = new Vector2(
				(float)((x - (DISP_X / 2)) / PPU),
				(float)(((DISP_Y / 2) - y) / PPU));
			po.goOuter.transform.localScale = new Vector2(
				(float)scale_x,
				(float)scale_y);
			po.goOuter.transform.localEulerAngles = new Vector3(
				0,
				0,
				(float)rot * -Mathf.Rad2Deg);
			return;	//ここまで。	忘れないで!!これを忘れると無条件に全ての候補を表示してしまうバグになる。
		}
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_draw_spr(ST_CbkQue pCbkQue, int pri, double x, double y, int[] aSprNo, int layer = 0/*Default*/, double scale_x = 1, double scale_y = 1, double rot = 0, double alpha = 1) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				draw_spr(x, y, aSprNo, scale_x, scale_y, rot, alpha);
			}, null);
	}
	//-----------------------------------------------------------------------------
	//スプライト描画(フォールバック無し)
	private static void draw_spr(double x, double y, int iSprNo, double scale_x, double scale_y, double rot, double alpha) {
		draw_spr(x, y, new int[] { iSprNo }, scale_x, scale_y, rot, alpha);
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_draw_spr(ST_CbkQue pCbkQue, int pri, double x, double y, int iSprNo, int layer = 0/*Default*/, double scale_x = 1, double scale_y = 1, double rot = 0, double alpha = 1) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				draw_spr(x, y, iSprNo, scale_x, scale_y, rot, alpha);
			}, null);
	}
	//-----------------------------------------------------------------------------
	//スプライトフォント描画
	private static void draw_sprf_fn(object user_data, int x, int y, double scale_x, double scale_y, int rgba, int iSprNo) {
		double alpha = (double)RGBA_GETALPHA(rgba) / (double)byte.MaxValue;
		draw_spr(x, y, iSprNo, scale_x, scale_y, 0, alpha);
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	private static void draw_sprf(VoidPtr pSprFnt, double x, double y, double scale_x, double scale_y, double ax, double ay, double alpha, string s) {
		SprFnt_Print(pSprFnt, draw_sprf_fn, null, (int)x, (int)y, scale_x, scale_y, ax, ay, D3DRGBA(1,1,1,alpha), s);
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_draw_sprf(ST_CbkQue pCbkQue, int pri, VoidPtr pSprFnt, double x, double y, double scale_x, double scale_y, double ax, double ay, double alpha, string s) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				draw_sprf(pSprFnt, x, y, scale_x, scale_y, ax, ay, alpha, s);
			}, null);
	}
	//-----------------------------------------------------------------------------
	//スプライトフォント描画(書式付き)
	private static void draw_sprf(VoidPtr pSprFnt, double x, double y, double scale_x, double scale_y, double ax, double ay, double alpha, string fmt, params object[] ap) {
		SprFnt_PrintF(pSprFnt, draw_sprf_fn, null, (int)x, (int)y, scale_x, scale_y, ax, ay, D3DRGBA(1,1,1,alpha), fmt, ap);
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_draw_sprf(ST_CbkQue pCbkQue, int pri, VoidPtr pSprFnt, double x, double y, double scale_x, double scale_y, double ax, double ay, double alpha, string fmt, params object[] ap) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				draw_sprf(pSprFnt, x, y, scale_x, scale_y, ax, ay, alpha, fmt, ap);
			}, null);
	}
	//-----------------------------------------------------------------------------
	//スプライトフォント描画(クロスフェード)
	private static void draw_sprf_xf(VoidPtr pSprFnt, double x, double y, double scale_x, double scale_y, double ax, double ay, double alpha, string s1, string s2, double transition/*0.0～1.0*/) {
		SprFnt_PrintXF(pSprFnt, draw_sprf_fn, null, (int)x, (int)y, scale_x, scale_y, ax, ay, D3DRGBA(1,1,1,alpha), s1, s2, transition);
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_draw_sprf_xf(ST_CbkQue pCbkQue, int pri, VoidPtr pSprFnt, double x, double y, double scale_x, double scale_y, double ax, double ay, double alpha, string s1, string s2, double transition/*0.0～1.0*/) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				draw_sprf_xf(pSprFnt, x, y, scale_x, scale_y, ax, ay, alpha, s1, s2, transition);
			}, null);
	}
	//-----------------------------------------------------------------------------
	private static void draw_line(double[,] xy, double w, int iSprNo, int rgba) {
		//Lineオブジェクトを取得する。
		LineObject lo = Graph.lineObjectPool.GetObject();
		//Lineオブジェクトを設定する。
		int n = xy.GetLength(0);
		lo.lineRenderer.numPositions = n;
		lo.lineRenderer.startWidth = lo.lineRenderer.endWidth = (float)(w / PPU);	//goInner.transform.localScaleはlineRenderer.startWidth,endWidthには影響しないようなので、lineRenderer.startWidth,endWidthにもスケールを掛ける必要が有ります。
		lo.lineRenderer.startColor = lo.lineRenderer.endColor = RGBA_GETCOLOR(rgba);
		Vector3[] positions = new Vector3[n];
		for(int i = 0; i < n; i++) {
			positions[i] = new Vector2(
				(float)xy[i,0/*x*/],
				(float)xy[i,1/*y*/]);
		}
		lo.lineRenderer.SetPositions(positions);
		//スプライト番号が指定されていたら…
		if(iSprNo != -1) {
			//スプライト情報を取得する。
			ST_SprDef stSprDef = GetSprDef(iSprNo);
			//ムービーならば…
			if(stSprDef.isMovie) {
				throw new ApplicationException();	//※未対応。多分、対応するのはそんなに難しくないと思う。CRIが返したマテリアルをlo.material.mainTextureに格納すれば良いだけなはずなので。
			//スプライトならば…
			} else {
				//スプライトを取得する。
				Sprite sprite = GetSprite(iSprNo);
				//Materialを設定する。
				lo.material.mainTexture = sprite.texture;
			}
		//スプライト番号が指定されていなければ…
		} else {
			//Materialを設定する。
			lo.material.mainTexture = null;
		}
	}
	//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	public static void CbkQue_draw_line(ST_CbkQue pCbkQue, int pri, double[,] xy, double w, int iSprNo, int rgba = -1) {
		CbkQue_Add(pCbkQue, pri, delegate(ST_CbkQue _pCbkQue , int _pri, object param) {
				draw_line(xy, w, iSprNo, rgba);
			}, null);
	}
	//-----------------------------------------------------------------------------
	//※TODO:ここに追加して下さい。
}
#endif//UNITY_5_3_OR_NEWER
